/*
  Top 10, a racing simulator
  Copyright (C) 2003-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "WheelProperties.hh"
#include "math/Plane.hh"

namespace top10
{
  namespace physX
  {
    const char *WheelProperties::NODE_NAME = "wheel_properties";

    WheelProperties::WheelProperties()
      : top10::util::XmlDumpable(NODE_NAME),
      m_radius(0.0),
      m_mass(0.0),
      m_inertia(0.0),
      m_inertia_y(0.0)
    {
    }

    top10::math::Vector
      WheelProperties::computeReaction(double load, top10::math::Vector down,
                                       double dist,
                                       top10::math::Vector speed) const
    {
      using top10::math::Vector;

      if (dist > m_radius)
	return Vector(0.0, 0.0, 0.0);

      const double stiffness = 1e4;
      double stiff_magnitude = (m_radius - dist) * stiffness;
      if (stiff_magnitude > 1e3)
	stiff_magnitude = 1e3;

      const double damping = 1e3;
      double damping_magnitude = 0.0;
      double down_speed = speed * down;
      if (down_speed > 0)
	damping_magnitude = down_speed * damping;
      if (damping_magnitude > 1e3)
	damping_magnitude = 1e3;

      return -(stiff_magnitude + damping_magnitude)*down;
    }

    top10::math::Vector
      WheelProperties::computeFriction(double load, top10::math::Vector down,
                                       double w_speed, top10::math::Vector w_dir,
				       top10::math::Vector speed,
				       FrictionData* out_data) const
    {
      using top10::math::Vector;

      if (load <= 0.0)
	return Vector(0.0, 0.0, 0.0);

      // A plane parallel to the ground.
      const top10::math::Plane ground(down, 0.0);

      // Projected speed on the ground
      const Vector flat_speed = top10::math::projectV(ground, speed);
      const double flat_speed_size = flat_speed.size();

      // velocity of the contact point on the wheel.
      const Vector patch_speed = m_radius*w_speed*(w_dir ^ down);

      // If the wheel is at rest, leave it alone.
      if (flat_speed_size < SMALL_VECTOR && patch_speed.size2() < SMALL_VECTOR*SMALL_VECTOR)
	return Vector(0.0, 0.0, 0.0);

      // unit vector parallel to the ground and perpendicular to the axis of the wheel
      Vector forward = -w_dir ^ down;
      double forward_size = forward.size();
      if (forward_size > SMALL_VECTOR)
	forward /= forward_size;
      else
	return Vector(0.0, 0.0, 0.0);  // Vehicle is busy tumbling over, it seems...

      // Side vector parallel to the ground
      Vector side = projectV(ground, w_dir);
      double side_size = side.size();
      if (side_size > SMALL_VECTOR)
	side /= side_size;
      else
	return Vector(0.0, 0.0, 0.0);  // Vehicle is busy tumbling over, it seems...

      // Projection of speed vectors on forward
      const double flat_speed_forward = flat_speed * forward;
      const double patch_speed_forward = patch_speed * forward;

      // Slip ratio
      double slip_ratio;
      if (fabs(flat_speed_forward) > 0.001)
	slip_ratio = -patch_speed_forward / flat_speed_forward -1.0;
      // speed too low, set a high friction ratio, otherwise the wheel may not get the initial grip it needs
      else if (-patch_speed_forward >= 0.0 == flat_speed_forward >= 0.0)
	slip_ratio = 1e15;
      else
	slip_ratio = -1e15;

      // Slip angle (the square of its sin() value, actually)
      double slip_angle;
      if (flat_speed_size > SMALL_VECTOR)
	slip_angle = ((forward ^ flat_speed) / flat_speed_size).size2();
      else
	slip_angle = 0.0;

      // Good for stats recording...
      if (out_data)
      {
	out_data->m_long_grip = getGripLong(slip_ratio);
	out_data->m_lat_grip = getGripLat(slip_angle);
	out_data->m_slip_ratio = slip_ratio;
	out_data->m_slip_angle = slip_angle;
      }

      Vector ret_long = forward*getGripLong(slip_ratio);
      if (flat_speed_forward < 0.0)
	ret_long = -ret_long;

      Vector ret_side = side*getGripLat(slip_angle);
      if (flat_speed * side < 0.0)
	ret_side = -ret_side;

      return load*(ret_long + ret_side);
    }

    double WheelProperties::getGripLong(double slip_factor) const
    {
      if (slip_factor < 0)
	return -m_grip_long.getY(-slip_factor);
      else
	return m_grip_long.getY(slip_factor);
    }

    double WheelProperties::getGripLat(double slip_angle) const
    {
      assert(slip_angle >= 0.0);
      assert(slip_angle <= 1.0);

      return m_grip_lat.getY(slip_angle);
    }

    void WheelProperties::setRadius(double radius)
    {
      m_radius = radius;
    }

    void WheelProperties::setInertia(double v)
    {
      m_inertia = v;
    }

    void WheelProperties::setMass(double v)
    {
      m_mass = v;
    }

    void WheelProperties::setInertiaVert(double v)
    {
      m_inertia_y = v;
    }

    void WheelProperties::insertGripLong(double slip_ratio, double grip)
    {
      m_grip_long.addXY(slip_ratio, grip);
    }

    void WheelProperties::insertGripLat(double slip_angle, double grip)
    {
      m_grip_lat.addXY(slip_angle, grip);
    }

    void WheelProperties::insertGripLatDegrees(double slip_angle, double grip)
    {
      double sin_val = sin(M_PI/180.0 *slip_angle);
      m_grip_lat.addXY(sin_val*sin_val, grip);
    }

    double WheelProperties::getRadius() const
    {
      return m_radius;
    }

    double WheelProperties::getMass() const
    {
      return m_mass;
    }

    double WheelProperties::getInertia() const
    {
      return m_inertia;
    }

    double WheelProperties::getInertiaVert() const
    {
      return m_inertia_y;
    }

    void WheelProperties::clearState()
    {
      m_radius = 0.0;
      m_mass = 0.0;
      m_inertia = 0.0;
      m_inertia_y = 0.0;
      m_grip_long = top10::math::Curve2D();
      m_grip_lat = top10::math::Curve2D();
    }

    int WheelProperties::loadXml(const TiXmlElement* node)
    {
      int status = 0;

      if (node->QueryDoubleAttribute("radius", &m_radius))
      {
	logXmlNodeError("Missing attribute 'radius'", node, top10::util::Log::Error);
	status = -1;
      }

      if (node->QueryDoubleAttribute("mass", &m_mass))
      {
	logXmlNodeError("Missing attribute 'mass'", node, top10::util::Log::Error);
	status = -1;
      }

      if (node->QueryDoubleAttribute("inertia", &m_inertia))
      {
	logXmlNodeError("Missing attribute 'inertia'", node, top10::util::Log::Error);
	status = -1;
      }

      if (node->QueryDoubleAttribute("inertia_y", &m_inertia_y))
      {
	logXmlNodeError("Missing attribute 'inertia_y'", node, top10::util::Log::Error);
	status = -1;
      }

      const TiXmlElement* child = node->FirstChildElement("grip_long");
      if (!child)
      {
	logXmlNodeError("Missing node 'grip_long'", node, top10::util::Log::Error);
	status = -1;
      }
      else
      {
	if (m_grip_long.loadXml(child))
	  status = -1;
      }

      child = node->FirstChildElement("grip_lat");
      if (!child)
      {
	logXmlNodeError("Missing node 'grip_lat'", node, top10::util::Log::Error);
	status = -1;
      }
      else
      {
	if (m_grip_lat.loadXml(child))
	  status = -1;
      }

      return status;
    }

    int WheelProperties::saveXml(TiXmlElement* node) const
    {
      node->SetDoubleAttribute("radius", m_radius);
      node->SetDoubleAttribute("mass", m_mass);
      node->SetDoubleAttribute("inertia", m_inertia);
      node->SetDoubleAttribute("inertia_y", m_inertia_y);
      
      TiXmlElement child("grip_long");
      m_grip_long.saveXml(&child);
      node->InsertEndChild(child);

      TiXmlElement child2("grip_lat");
      m_grip_lat.saveXml(&child2);
      node->InsertEndChild(child2);

      return 0;
    }
  }
}
