#include "SetupSettings.hh"
#include <algorithm>


namespace top10
{
  namespace physX
  {
    typedef top10::util::XmlDumpable XmlT;


    /*
     * TyreSetting
     */

    const char* TyreSetting::s_nodename = "tyre_setting";

    TyreSetting::TyreSetting()
      : XmlT(s_nodename)
    {
      clearState();
    }



    TyreSetting::TyreSetting(double side_k, double long_k, double breadth,
			     const char* desc)
      : XmlT(s_nodename),
	m_side_k(side_k),
	m_long_k(long_k),
	m_breadth(breadth),
	m_desc(desc)
    {
    }



    void TyreSetting::clearState()
    {
      m_side_k = 0.0;
      m_long_k = 0.0;
      m_breadth = 0.0;
      m_desc = "";
    }



    int TyreSetting::loadXml(const TiXmlElement* el)
    {
      el->QueryDoubleAttribute("side", &m_side_k);
      el->QueryDoubleAttribute("long", &m_long_k);
      el->QueryDoubleAttribute("breadth", &m_breadth);
      const char* desc = el->Attribute("desc");
      if (desc)
	m_desc = desc;

      return 0;
    }



    int TyreSetting::saveXml(TiXmlElement* el) const
    {
      el->SetDoubleAttribute("side", m_side_k);
      el->SetDoubleAttribute("long", m_long_k);
      el->SetDoubleAttribute("breadth", m_breadth);
      el->SetAttribute("desc", m_desc);

      return 0;
    }



    /*
     * EngineSetting
     */

    const char* EngineSetting::s_nodename = "engine_setting";

    EngineSetting::EngineSetting()
      : XmlT(s_nodename)
    {
      clearState();
    }



    EngineSetting::EngineSetting(double opt,
				 const char* desc)
      : XmlT(s_nodename),
	m_opt(opt),
	m_desc(desc)
    {
    }



    void EngineSetting::clearState()
    {
      m_opt = 0.0;
      m_desc = "";
    }



    int EngineSetting::loadXml(const TiXmlElement* el)
    {
      el->QueryDoubleAttribute("opt", &m_opt);
      const char* desc = el->Attribute("desc");
      if (desc)
	m_desc = desc;

      return 0;
    }



    int EngineSetting::saveXml(TiXmlElement* el) const
    {
      el->SetDoubleAttribute("opt", m_opt);
      el->SetAttribute("desc", m_desc);

      return 0;
    }


    /*
     * StiffSetting
     */

    const char* StiffSetting::s_nodename = "stiff_setting";

    StiffSetting::StiffSetting()
      : XmlT(s_nodename)
    {
      clearState();
    }



    StiffSetting::StiffSetting(double stiff,
			       const char* desc)
      : XmlT(s_nodename),
	m_stiff(stiff),
	m_desc(desc)
    {
    }



    void StiffSetting::clearState()
    {
      m_stiff = 0.0;
      m_desc = "";
    }



    int StiffSetting::loadXml(const TiXmlElement* el)
    {
      el->QueryDoubleAttribute("stiff", &m_stiff);
      const char* desc = el->Attribute("desc");
      if (desc)
	m_desc = desc;

      return 0;
    }



    int StiffSetting::saveXml(TiXmlElement* el) const
    {
      el->SetDoubleAttribute("stiff", m_stiff);
      el->SetAttribute("desc", m_desc);

      return 0;
    }



    /*
     * Settings
     */
    const char * Settings::s_nodename = "setup_settings";

    Settings::Settings():
      XmlT(s_nodename)
    {
      clearState();
    }
     


    void Settings::clearState()
    {
      m_tyre_settings.clear();
      m_engine_settings.clear();
      m_stiff_settings.clear();
    }



    int Settings::loadXml(const TiXmlElement* el)
    {
      const TiXmlElement* child = el->FirstChildElement();
      while (child)
      {
	TyreSetting tyre;
	EngineSetting engine;
	StiffSetting stiff;

	if (tyre.handles(child))
	{
	  tyre.load(child);
	  m_tyre_settings.push_back(tyre);
	}
	else if (engine.handles(child))
	{
	  engine.load(child);
	  m_engine_settings.push_back(engine);
	}
	else if (stiff.handles(child))
	{
	  stiff.load(child);
	  m_stiff_settings.push_back(stiff);
	}
	else
	  top10::util::Log::getSingle()->send(
	    top10::util::Log::Warning,
	    "Settings/loadXml",
	    std::string("Skipped node ") + child->Value() );

	child = child->NextSiblingElement();
      }

      int status = 0;

      if (m_tyre_settings.size() == 0)
      {
	top10::util::Log::getSingle()->send(
	  top10::util::Log::Error,
	  "Settings/loadXml",
	  "No tyre setting found" );
	status = -1;
      }

      if (m_engine_settings.size() == 0)
      {
	top10::util::Log::getSingle()->send(
	  top10::util::Log::Error,
	  "Settings/loadXml",
	  "No engine setting found" );
	status = -2;
      }

      if (m_stiff_settings.size() == 0)
      {
	top10::util::Log::getSingle()->send(
	  top10::util::Log::Error,
	  "Settings/loadXml",
	  "No chassis stiffness setting found" );
	status = -3;
      }

      return status;
    }



    int Settings::saveXml(TiXmlElement* el) const
    {
      top10::util::SaveInserterFunction saver(el);

      std::for_each(m_tyre_settings.begin(), m_tyre_settings.end(), saver);
      std::for_each(m_engine_settings.begin(), m_engine_settings.end(), saver);
      std::for_each(m_stiff_settings.begin(), m_stiff_settings.end(), saver);

      return 0;
    }




    /*
     * SettingsState
     */
    SettingsState::SettingsState()
    {
      for (unsigned int i=COMP_FIRST; i <= COMP_LAST; ++i)
	m_idxs[i] = 0;
    }



    std::string
    SettingsState::getDesc(Components c) const
    {
      switch (c)
      {
      case COMP_FRONT_TYRE_STICKYNESS:
      case COMP_REAR_TYRE_STICKYNESS:
	return m_settings->m_tyre_settings.at(m_idxs[c]).m_desc;
	
      case COMP_ENGINE_USAGE:
	return m_settings->m_engine_settings.at(m_idxs[c]).m_desc;

      case COMP_FRONT_STIFFNESS:
      case COMP_REAR_STIFFNESS:
	return m_settings->m_stiff_settings.at(m_idxs[c]).m_desc;
      }

      return "???";
    }



    unsigned int
    SettingsState::getNumSettings(Components c) const
    {
      unsigned int num_vals = 0;

      switch (c)
      {
      case COMP_FRONT_TYRE_STICKYNESS:
	num_vals = m_settings->m_tyre_settings.size();
	break;

      case COMP_REAR_TYRE_STICKYNESS:
	num_vals = m_settings->m_tyre_settings.size();
	break;

      case COMP_ENGINE_USAGE:
	num_vals = m_settings->m_engine_settings.size();
	break;

      case COMP_FRONT_STIFFNESS:
	num_vals = m_settings->m_stiff_settings.size();
	break;

      case COMP_REAR_STIFFNESS:
	num_vals = m_settings->m_stiff_settings.size();
	break;
      }

      return num_vals;
    }



    void SettingsState::incr(Components c)
    {
      ++m_idxs[(unsigned int)c];
      m_idxs[(unsigned int)c] %= getNumSettings(c);
    }



    void SettingsState::decr(Components c)
    {
      if (m_idxs[(unsigned int)c] == 0)
	m_idxs[(unsigned int)c] = getNumSettings(c) -1;
      else
	--m_idxs[(unsigned int)c];
    }



    std::string getStr(SettingsState::Components c)
    {
      typedef SettingsState T;

      switch (c)
      {
      case T::COMP_FRONT_TYRE_STICKYNESS:
	return "Front tyres";

      case T::COMP_REAR_TYRE_STICKYNESS:
	return "Rear tyres";

      case T::COMP_ENGINE_USAGE:
	return "Engine power curve";

      case T::COMP_FRONT_STIFFNESS:
	return "Front stiffness";

      case T::COMP_REAR_STIFFNESS:
	return "Rear stiffness";
      }

      return "???";
    }
  }
}
