/*
  Top10, a racing simulator
  Copyright (C) 2000-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#ifndef TOP10_RIGID_BODY_STATE_HH
#define TOP10_RIGID_BODY_STATE_HH

#include "RigidBodyProperties.hh"
#include "math/Matrix.hh"

namespace top10
{
  namespace physX
  {
    class RigidBodyState: public top10::util::RefCount
    {
    public:
      RigidBodyState(const RigidBodyProperties*);

      inline const RigidBodyProperties* getProperties() const { return m_props.getPtr(); }

      //! Set the orientation of this body.
      /*!
	\param M Orientation matrix, its columns denote vectors e0, e1, e2 which must have length 1.
	         It must also be the case that e(n) ^ e(n+1) = e(n+2)
      */
      void setOrient(const top10::math::Matrix3& M);

      //! Set the translation of this body.
      void setTranslation(const top10::math::Vector& v);

      //! Clear all forces and torqies previously applied.
      void clearForces();

      //! Apply a constant force.
      /*!
	\param pos Where the force is applied, in the global frame.
	\param force The force to apply, in the global frame.
	*/
      void applyForceAt(top10::math::Vector pos, top10::math::Vector force);
      
      //! Aply a constant torque.
      /*!
	\param torque Torque to apply, in the global frame.
      */
      void applyTorque(top10::math::Vector torque);

      //! Do a simulation step.
      /*!
	The simulation step uses the forces and torques applied previously.
      */
      void integrate(double t, double dt);

      inline top10::math::Vector         getTranslation() const { return m_translation; }
      inline const top10::math::Matrix3& getOrient()      const { return m_orient; }
      inline const top10::math::Matrix3& getOrientInv()   const { return m_orient_inv; }
      inline top10::math::Vector         getSpeed()       const { return m_speed; }
      inline top10::math::Vector         getWSpeed()      const { return m_angular_speed; }
      inline top10::math::Vector         getForce()       const { return m_force; }
      inline top10::math::Vector         getTorque()      const { return m_torque; }

      //! Return the center of mass
      inline top10::math::Vector getCenter() const { return m_props->getCenter() + m_translation; }

      //! Return the global coordinates of a vertex in the local frame.
      top10::math::Vector localToGlobal(top10::math::Vector) const;

      //! Return the speed of a vertex
      /*!
      \param pos Position of the vertex in the local frame.
      \return Speed in the global frame.
      */
      top10::math::Vector getSpeedAtL(top10::math::Vector) const;

      //! Return the total mass of the rigid body.
      double getMass() const;

    private:

      top10::util::Ref<const RigidBodyProperties> m_props;

      top10::math::Vector m_translation;

      top10::math::OrthoNorm3 m_orient;

      top10::math::OrthoNorm3 m_orient_inv;

      //! Linear speed, in the global frame.
      top10::math::Vector m_speed;

      //! Angular speed, in the local frame (?).
      top10::math::Vector m_angular_speed;

      //! Force, in the global frame.
      top10::math::Vector m_force;

      //! Torque, in the global frame.
      top10::math::Vector m_torque;
    };

    typedef top10::util::Ref< RigidBodyState > RigidBodyStateRef;
    typedef std::vector< RigidBodyStateRef > RigidBodyStateRefs;
  }
}

#endif
