/*
  Top10, a racing simulator
  Copyright (C) 2000-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "RigidBodyState.hh"
#include "Integrator.hh"

namespace top10
{
  namespace physX
  {
    RigidBodyState::RigidBodyState(const RigidBodyProperties* props)
      : m_props(props)
    {
    }

    double RigidBodyState::getMass() const
    {
      return m_props->getMass();
    }

    void RigidBodyState::setOrient(const top10::math::Matrix3& M)
    {
      m_orient = M;
      m_orient_inv = m_orient.transpose();
    }

    void RigidBodyState::setTranslation(const top10::math::Vector& v)
    {
      m_translation = v;
    }

    void RigidBodyState::clearForces()
    {
      m_force = top10::math::Vector(0.0, 0.0, 0.0);
      m_torque = top10::math::Vector(0.0, 0.0, 0.0);
    }

    void RigidBodyState::applyForceAt(top10::math::Vector pos, top10::math::Vector force)
    {
      m_force += force;
      m_torque += (pos - getCenter()) ^ force;
    }

    void RigidBodyState::applyTorque(top10::math::Vector torque)
    {
      m_torque += torque;
    }

    void RigidBodyState::integrate(double t, double dt)
    {
      // Linear movement
      ConstantVectorAccel accel(m_force / m_props->getMass());
      integrateDirect(m_translation, m_speed, t, dt, accel);

      // Angular movement
      top10::math::Matrix3 M = m_props->getInertiaInv() * m_orient_inv;
      top10::math::Vector w_accel = M * m_torque;
      ConstantAngularAccel angular_accel(w_accel);
      integrateAngular(m_orient, m_angular_speed, t, dt, angular_accel);
      m_orient.correct();
      m_orient_inv = m_orient.transpose();
    }

    top10::math::Vector RigidBodyState::localToGlobal(top10::math::Vector p) const
    {
      return m_orient*(p-m_props->getCenter()) + getCenter();
    }

    top10::math::Vector RigidBodyState::getSpeedAtL(top10::math::Vector pos) const
    {
      return m_speed + m_orient*((m_props->getCenter()-pos)^m_angular_speed);
    }

  }
}
