/*
  Top10, a racing simulator
  Copyright (C) 2000-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#ifndef TOP10_RIGID_BODY_PROPERTIES_H
#define TOP10_RIGID_BODY_PROPERTIES_H

#include "math/Matrix.hh"
#include "util/XmlDumpable.hh"
#include "util/RefCount.hh"

namespace top10
{
  namespace physX
  {

    //! Non-mutable properties of a rigid body.
    class RigidBodyProperties: public top10::util::XmlDumpable,
                               public top10::util::RefCount
    {
    public:
      static const char* NODE_NAME;
      
    public:
      //! Body with mass 0.
      RigidBodyProperties();

      void getMasses(std::vector< top10::math::Vector >* positions, std::vector< double >* masses) const;

      //! Add a mass at a given location.
      /*!
	\param pos Position of the mass in the body's frame.
	\param mass Mass in Kg. Must be strictly positive.
      */
      void addMass(top10::math::Vector pos, double mass);

      //! Compute m_inertia, m_inertia_inv, m_center and m_mass.
      void update();

      // Read-only accessors.
      top10::math::Vector  getCenter()     const;
      double               getMass()       const;
      top10::math::Matrix3 getInertia()    const;
      top10::math::Matrix3 getInertiaInv() const;

    protected:
      int loadXml(const TiXmlElement*);
      int saveXml(TiXmlElement*) const;
      void clearState();

    private:
      //! A mass and a location.
      struct VertexMass
      {
	top10::math::Vector m_pos;
	double              m_mass;
      };

      //! All masses with their locations in the body's frame.
      std::vector<VertexMass> m_masses;

      //! Inertia matrix.
      top10::math::Matrix3    m_inertia;

      //! Inverse of the inertia matrix.
      top10::math::Matrix3    m_inertia_inv;

      //! Center of mass.
      top10::math::Vector     m_center;

      //! Total mass.
      double                  m_mass;
    };

    typedef top10::util::Ref< RigidBodyProperties > RigidBodyPropertiesRef;
    typedef std::vector< RigidBodyPropertiesRef > RigidBodyPropertiesRefs;

    typedef top10::util::Ref< const RigidBodyProperties > RigidBodyPropertiesConstRef;
    typedef std::vector< RigidBodyPropertiesConstRef > RigidBodyPropertiesConstRefs;
  }
}

#endif
