/*
  Top10, a racing simulator
  Copyright (C) 2000-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "RigidBodyProperties.hh"

namespace top10
{
  namespace physX
  {
    const char* RigidBodyProperties::NODE_NAME = "rigid_body_properties";

    RigidBodyProperties::RigidBodyProperties()
      : top10::util::XmlDumpable(NODE_NAME),
      top10::util::RefCount(),
      m_masses(),
      m_inertia(),
      m_inertia_inv(),
      m_center(0.0, 0.0, 0.0),
      m_mass(0.0)
    {
    }

    void RigidBodyProperties::clearState()
    {
      m_masses.clear();
    }

    int RigidBodyProperties::loadXml(const TiXmlElement* node)
    {
      int status = 0;

      const TiXmlElement* child = node->FirstChildElement("mass");
      while (child)
      {
	top10::math::Vector v;
	try {
	  v.loadXml(child);
	  double mass = 0.0;
	  int status2 = child->QueryDoubleAttribute("mass", &mass);
	  if (status2 == 0)
	    addMass(v, mass);
	  else
	  {
	    logXmlNodeError("Missing or incorrect attribute 'mass'", child, top10::util::Log::Error);
	    status = status2;
	  }
	}
	catch (const std::string& err)
	{
	  logXmlNodeError(err, child, top10::util::Log::Error);
	  status = -1;
	}
	child = child->NextSiblingElement("mass");
      }

      if (!m_masses.empty())
	update();
      else
      {
	logXmlNodeError("Rigid body has no mass", node, top10::util::Log::Error);
	status = -1;
      }

      return status;
    }

    int RigidBodyProperties::saveXml(TiXmlElement* node) const
    {
      for (std::vector<VertexMass>::const_iterator it = m_masses.begin();
	it != m_masses.end();
	++it)
      {
	TiXmlElement child("mass");
	child.SetDoubleAttribute("mass", it->m_mass);
	it->m_pos.saveXml(&child);
	node->InsertEndChild(child);
      }

      return 0;
    }

    void RigidBodyProperties::addMass(top10::math::Vector pos, double mass)
    {
      assert(mass >= 0.0);
      
      if (mass == 0.0)
	return;

      VertexMass new_one;
      new_one.m_pos = pos;
      new_one.m_mass = mass;
      m_masses.push_back(new_one);
    }

    void RigidBodyProperties::update()
    {
      m_center = top10::math::Vector(0.0, 0.0, 0.0);
      m_mass = 0.0;

      // Compute mass center of the body
      for (std::vector<VertexMass>::const_iterator it = m_masses.begin();
	it != m_masses.end();
	++it)
      {
	m_center += it->m_mass * it->m_pos;
	m_mass += it->m_mass;
      }

      assert(m_mass > 0.0);
      m_center /= m_mass;

      // Compute matrix of inertia
      double Ixx=0, Iyy=0, Izz=0, Ixy=0, Ixz=0, Iyz=0;
      for (std::vector<VertexMass>::const_iterator it = m_masses.begin();
	it != m_masses.end();
	++it)
      {
	  double x = it->m_pos.x - m_center.x;
	  double y = it->m_pos.y - m_center.y;
	  double z = it->m_pos.z - m_center.z;

	  Ixx += it->m_mass*(y*y + z*z);
	  Iyy += it->m_mass*(x*x + z*z);
	  Izz += it->m_mass*(x*x + y*y);
	  Ixy += it->m_mass*(x*y);
	  Ixz += it->m_mass*(x*z);
	  Iyz += it->m_mass*(y*z);
      }

      double vals[3][3] = { 
	{ Ixx, -Ixy, -Ixz },
	{-Ixy,  Iyy, -Iyz },
	{-Ixz, -Iyz,  Izz }
      };
      m_inertia = top10::math::Matrix3(vals);
      m_inertia_inv = top10::math::inverse(m_inertia);
    }

    top10::math::Vector
      RigidBodyProperties::getCenter() const
    {
      return m_center;
    }

    double RigidBodyProperties::getMass() const
    {
      return m_mass;
    }

    top10::math::Matrix3
      RigidBodyProperties::getInertia() const
    {
      return m_inertia;
    }

    top10::math::Matrix3
      RigidBodyProperties::getInertiaInv() const
    {
      return m_inertia_inv;
    }
  }
}
