/*
  Top 10, a racing simulator
  Copyright (C) 2003-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "LinearSpringProperties.hh"

namespace top10
{
  namespace physX
  {
    const char* LinearSpringProperties::NODE_NAME = "lin_spring_properties";

    LinearSpringProperties::LinearSpringProperties()
      : top10::util::XmlDumpable(NODE_NAME),
      m_length(0.0),
      m_stiffness(0.0),
      m_damping(0.0)
    {
    }

    void LinearSpringProperties::setStiffness(double v)
    {
      m_stiffness = v;
    }

    void LinearSpringProperties::setDamping(double v)
    {
      m_damping = v;
    }

    void LinearSpringProperties::setLength(double v)
    {
      m_length = v;
    }

    top10::math::Vector
      LinearSpringProperties::getForce(top10::math::Vector diff,
				       top10::math::Vector speed) const
    {
      top10::math::Vector dir = diff;
      double diff_size = dir.size();

      if (diff_size > SMALL_VECTOR)
	dir /= diff_size;
      else return top10::math::Vector(0.0, 0.0, 0.0);

      return (m_stiffness * (m_length - diff_size)
	    - m_damping * (speed * dir)) * dir;
    }

    int LinearSpringProperties::loadXml(const TiXmlElement* node)
    {
      int status = 0;

      if (node->QueryDoubleAttribute("length", &m_length))
      {
	logXmlNodeError("Missing attribute 'length'", node, top10::util::Log::Error);
	status = -1;
      }

      if (node->QueryDoubleAttribute("stiff", &m_stiffness))
      {
	logXmlNodeError("Missing attribute 'stiff'", node, top10::util::Log::Error);
	status = -1;
      }

      if (node->QueryDoubleAttribute("damp", &m_damping))
      {
	logXmlNodeError("Missing attribute 'damp'", node, top10::util::Log::Error);
	status = -1;
      }

      return status;
    }

    int LinearSpringProperties::saveXml(TiXmlElement* node) const
    {
      node->SetDoubleAttribute("length", m_length);
      node->SetDoubleAttribute("stiff", m_stiffness);
      node->SetDoubleAttribute("damp", m_damping);
      return 0;
    }

    void LinearSpringProperties::clearState()
    {
      m_length = 0.0;
      m_stiffness = 0.0;
      m_damping = 0.0;
    }

  }
}