/*
  Top 10, a racing simulator
  Copyright (C) 2003-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "KartState.hh"
#include "Integrator.hh"

namespace top10
{
  namespace physX
  {

    KartState::KartState(const KartProperties* props)
      : m_engine(props->getEngine()),
      m_gear_box(props->getGearBox()),
      m_clutch(props->getClutch()),
      m_axle(props->getAxle())
    {
      top10::util::RefArray<const RigidBodyProperties> body_props;
      props->getBodies(&body_props);

      for (int i=0; i<body_props.size(); ++i)
      {
	RigidBodyState* body = new RigidBodyState(body_props[i]);
	m_bodies.push_back(body);
      }

      m_wheels.reserve(4);
      for (int i=0; i<2; ++i)
      {
	WheelState wheel(props->getFrontWheels());
	m_wheels.push_back(wheel);
      }

      for (int i=2; i<4; ++i)
      {
	WheelState wheel(props->getRearWheels());
	m_wheels.push_back(wheel);
      }

      top10::util::RefArray<const LinearSpringProperties> lin_springs;
      props->getLinearSprings(&lin_springs);

      top10::util::RefArray<const AngularSpringProperties> ang_springs;
      props->getAngularSprings(&ang_springs);

      std::vector<SpringAttachment> lin_attach;
      props->getLinearAttachments(&lin_attach);

      for (unsigned int i=0; i<lin_attach.size(); ++i)
      {
	const SpringAttachment& att(lin_attach[i]);
	LinearSpring spring;
	spring.setProperties(lin_springs[att.m_spring_idx]);
	spring.setBody1(m_bodies[ att.m_idx_body[0] ], att.m_vec[0]);
	spring.setBody2(m_bodies[ att.m_idx_body[1] ], att.m_vec[1]);
	
	m_lin_springs.push_back(new LinearSpring(spring));
      }

      props->getWheelLinearAttachments(&lin_attach);
      for (unsigned int i=0; i<lin_attach.size(); ++i)
      {
	const SpringAttachment& att(lin_attach[i]);
	LinearSpring spring;
	spring.setProperties(lin_springs[att.m_spring_idx]);
	
	if (att.m_idx_body[0] < 0 || att.m_idx_body[0] >= 4)
	  top10::util::Log::getSingle()->send(top10::util::Log::Error, "KartState", "Wheel index out of range");
	else
	{
	  spring.setBody1(m_wheels[ att.m_idx_body[0] ].getRigidBody(), att.m_vec[0]);
	  spring.setBody2(m_bodies[ att.m_idx_body[1] ], att.m_vec[1]);
	
	  m_lin_springs.push_back(new LinearSpring(spring));
	}
      }

      std::vector<SpringAttachment> ang_attach;
      props->getAngularAttachments(&ang_attach);

      for (unsigned int i=0; i<ang_attach.size(); ++i)
      {
	const SpringAttachment& att(ang_attach[i]);
	AngularSpring spring;
	spring.setProperties(ang_springs[ att.m_spring_idx ]);
	spring.setBody1(m_bodies[ att.m_idx_body[0] ], att.m_vec[0]);
	spring.setBody2(m_bodies[ att.m_idx_body[1] ], att.m_vec[1]);
	
	m_ang_springs.push_back(new AngularSpring(spring));
      }

      props->getWheelAngularAttachments(&ang_attach);
      for (unsigned int i=0; i<ang_attach.size(); ++i)
      {
	const SpringAttachment& att(ang_attach[i]);
	AngularSpring spring;
	spring.setProperties(ang_springs[ att.m_spring_idx ]);
	
	if (att.m_idx_body[0] < 0 || att.m_idx_body[0] >= 4)
	  top10::util::Log::getSingle()->send(top10::util::Log::Error, "KartState", "Wheel index out of range");
	else
	{
	  spring.setBody1(m_wheels[ att.m_idx_body[0] ].getRigidBody(), att.m_vec[0]);
	  spring.setBody2(m_bodies[ att.m_idx_body[1] ], att.m_vec[1]);
	
	  m_ang_springs.push_back(new AngularSpring(spring));
	}
      }
    }


    void KartState::setClutch(bool b)
    {
      m_clutch.toggle(b);
    }

    void KartState::setThrottle(double v)
    {
      m_engine.setThrottle(v);
    }

    void KartState::setGear(int g)
    {
      m_gear_box.setGear(g);
    }

    EngineState KartState::getEngine() const
    {
      return m_engine;
    }

    GearBoxState KartState::getGearBox() const
    {
      return m_gear_box;
    }

    ClutchState KartState::getClutch() const
    {
      return m_clutch;
    }

    AxleState KartState::getAxle() const
    {
      return m_axle;
    }

    void KartState::updateDriveTrain(double t, double dt, EngineState* new_engine, AxleState* new_axle)
    {
      double accel = 0.0;

      // Update the engine state
      if (m_gear_box.isCoupled())
      {
	double R = m_gear_box.getRatio();
	assert(fabs(R) > 0.001);
	double r = 1.0 / R;

	double factor = R*m_axle.getInertia() + r*m_engine.getInertia();
	double clutch_torque = m_clutch.getFriction()*(m_clutch.getWSpeed() - m_axle.getWSpeed());
	double engine_torque = r*m_engine.getTorque();
	accel = -(clutch_torque - engine_torque)/factor;
      }
      else
      {
	accel = m_engine.getTorque() / m_engine.getInertia();
      }

      DoubleState state;
      state.setPos(0.0);
      state.setSpeed(m_engine.getWSpeed());

      ConstantDoubleAccel const_accel(accel);
      top10::physX::integrate<DoubleState, DoubleDerivative, double, ConstantDoubleAccel>(state, t, dt, const_accel);
      if (state.getSpeed() > 0.0)
	new_engine->setWSpeed(state.getSpeed());
      else
	new_engine->setWSpeed(0.0);

      // Update the axle state
      if (m_gear_box.isCoupled())
      {
	new_axle->setWSpeed(-m_gear_box.getRatio()*new_engine->getWSpeed());
      }
      else
      {
	double accel = m_clutch.getFriction()*(m_clutch.getWSpeed() - m_axle.getWSpeed()) / m_axle.getInertia();

	DoubleState state;
	state.setPos(0.0);
	state.setSpeed(m_axle.getWSpeed());

	ConstantDoubleAccel const_accel(accel);
	top10::physX::integrate<DoubleState, DoubleDerivative, double, ConstantDoubleAccel>(state, t, dt, const_accel);
	new_axle->setWSpeed(state.getSpeed());
      }
    }
  }
}
