/*
  Top 10, a racing simulator
  Copyright (C) 2003-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#ifndef TOP10_KART_PROPERTIES_H
#define TOP10_KART_PROPERTIES_H

#include "EngineProperties.hh"
#include "GearBoxProperties.hh"
#include "ClutchProperties.hh"
#include "AxleProperties.hh"
#include "RigidBodyProperties.hh"
#include "WheelProperties.hh"
#include "LinearSpringProperties.hh"
#include "AngularSpringProperties.hh"

#include "util/XmlDumpable.hh"

namespace top10
{
  namespace physX
  {



    //! Describes an spring (linear or angular) attached to two rigid bodies.
    struct SpringAttachment
    {
      SpringAttachment(): m_spring_idx(0) { m_idx_body[0] = m_idx_body[1] = 0; }

      //! Index of the spring properties in KartProperties.
      int m_spring_idx;

      //! Indices of the two rigid bodies in KartProperties.
      int m_idx_body[2];

      //! Location of the attach points for a linear spring, or directions for an angular spring.
      top10::math::Vector m_vec[2];

      int loadXml(const TiXmlElement*);
      int saveXml(TiXmlElement*) const;
    };




    //! Properties of a go-kart.
    class KartProperties : public top10::util::RefCount, public top10::util::XmlDumpable
    {
    public:
      static const char* NODE_NAME;

      // Indices inside m_wheel_attach.
      static const unsigned int FL = 0; //<! Front left
      static const unsigned int FR = 1; //<1 Front right
      static const unsigned int RL = 2; //<! Rear left
      static const unsigned int RR = 3; //<! Rear right

    public:
      //! Default properties. Must be initialized to be usable.
      KartProperties();

      // Read-only accessors.
      const EngineProperties* getEngine() const;
      const GearBoxProperties* getGearBox() const;
      const ClutchProperties* getClutch() const;
      const AxleProperties* getAxle() const;
      const WheelProperties* getFrontWheels() const;
      const WheelProperties* getRearWheels() const;

      void getBodies(RigidBodyPropertiesConstRefs* bodies) const;
      void getLinearAttachments(std::vector<SpringAttachment>* springs) const;
      void getAngularAttachments(std::vector<SpringAttachment>* springs) const;
      void getLinearSprings(LinearSpringPropertiesConstRefs* springs) const;
      void getAngularSprings(AngularSpringPropertiesRefs* springs) const;
      
      void getWheelLinearAttachments(std::vector<SpringAttachment>*) const;
      void getWheelAngularAttachments(std::vector<SpringAttachment>*) const;

      // Reset to default values.
      void clearState();

    protected:
      int loadXml(const TiXmlElement*);
      int saveXml(TiXmlElement*) const;

    private:
      top10::util::Ref<EngineProperties> m_engine;
      top10::util::Ref<GearBoxProperties> m_gear_box;
      top10::util::Ref<ClutchProperties> m_clutch;
      top10::util::Ref<AxleProperties> m_axle;
      top10::util::Ref<WheelProperties> m_front_wheels;
      top10::util::Ref<WheelProperties> m_rear_wheels;

      RigidBodyPropertiesRefs m_bodies;
      LinearSpringPropertiesRefs m_lin_spring_specs;
      AngularSpringPropertiesRefs m_ang_spring_specs;

      std::vector<SpringAttachment> m_lin_springs;
      std::vector<SpringAttachment> m_ang_springs;

      //! Attachment of the wheel on the body using a linear spring.
      /*!
	First body denotes the index of the wheel, should be between 1 and 4.
	Second body refers to a body in m_bodies.
	*/
      std::vector<SpringAttachment> m_wheel_lin_attach;

      //! Attachment of the wheel on the body using an angular spring.
      /*!
	First body denotes the index of the wheel, should be between 1 and 4.
	Second body refers to a body in m_bodies.
	*/
      std::vector<SpringAttachment> m_wheel_ang_attach;
    };
  }
}

#endif
