/*
  Top 10, a racing simulator
  Copyright (C) 2003-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "KartProperties.hh"
#include "util/strconv.hh"

namespace top10
{
  namespace physX
  {
    const char* KartProperties::NODE_NAME = "kart_properties";




    KartProperties::KartProperties()
      : top10::util::XmlDumpable(NODE_NAME)
    {
      m_engine = new EngineProperties;
      m_gear_box = new GearBoxProperties;
      m_clutch = new ClutchProperties;
      m_axle = new AxleProperties;
      m_front_wheels = new WheelProperties;
      m_rear_wheels = new WheelProperties;
    }




    const EngineProperties* KartProperties::getEngine() const
    {
      return m_engine.getPtr();
    }




    const GearBoxProperties* KartProperties::getGearBox() const
    {
      return m_gear_box.getPtr();
    }




    const ClutchProperties* KartProperties::getClutch() const
    {
      return m_clutch.getPtr();
    }




    const AxleProperties* KartProperties::getAxle() const
    {
      return m_axle.getPtr();
    }




    int KartProperties::loadXml(const TiXmlElement* node)
    {
      int status = 0;
      
      // Load the engine
      const TiXmlElement* child = node->FirstChildElement(EngineProperties::NODE_NAME);
      if (child == 0)
      {
	status = -1;
	logXmlNodeError("Missing engine_properties node", node, top10::util::Log::Error);
      }
      else
      {
	EngineProperties* new_engine = new EngineProperties();
	new_engine->load(child);
	m_engine = new_engine;
      }

      // load the gear box
      child = node->FirstChildElement(GearBoxProperties::NODE_NAME);
      if (child == 0)
      {
	status = -1;
	logXmlNodeError("Missing gearbox_properties node", node, top10::util::Log::Error);
      }
      else
      {
	GearBoxProperties* new_box = new GearBoxProperties();
	new_box->load(child);
	m_gear_box = new_box;
      }

      // Load the clutch
      child = node->FirstChildElement(ClutchProperties::NODE_NAME);
      if (child == 0)
      {
	status = -1;
	logXmlNodeError("Missing clutch_properties node", node, top10::util::Log::Error);
      }
      else
      {
	ClutchProperties* new_clutch = new ClutchProperties();
	new_clutch->load(child);
	m_clutch = new_clutch;
      }

      // load the axle
      child = node->FirstChildElement(AxleProperties::NODE_NAME);
      if (child == 0)
      {
	status = -1;
	logXmlNodeError("Missing axle_properties node", node, top10::util::Log::Error);
      }
      else
      {
	AxleProperties* new_axle = new AxleProperties();
	new_axle->load(child);
	m_axle = new_axle;
      }

      // Load the front and rear wheels.
      bool has_front = false;
      bool has_rear = false;
      child = node->FirstChildElement(WheelProperties::NODE_NAME);
      while (child)
      {
	const char* location = child->Attribute("location");
	if (location && std::string(location) == "front")
	{
	  has_front = true;
	  m_front_wheels->load(child);
	}
	else if (location && std::string(location) == "rear")
	{
	  has_rear = true;
	  m_rear_wheels->load(child);
	}

	child = child->NextSiblingElement(child->Value());
      }

      if (!has_front)
	logXmlNodeError("Missing front wheel spec", node);
      if (!has_rear)
	logXmlNodeError("Missing rear wheel spec", node);

      // Load all bodies
      child = node->FirstChildElement(RigidBodyProperties::NODE_NAME);
      while (child)
      {
	int idx = 0;
	child->Attribute("index", &idx);
	if (m_bodies[idx])
	  logXmlNodeError("Duplicate rigid body index "+top10::util::toString(idx), child);
	m_bodies[idx] = new RigidBodyProperties;
	m_bodies[idx]->load(child);

	child = child->NextSiblingElement(child->Value());
      }

      // Load all linear springs
      child = node->FirstChildElement(LinearSpringProperties::NODE_NAME);
      while (child)
      {
	int idx = 0;
	child->Attribute("index", &idx);
	if (m_lin_spring_specs[idx])
	  logXmlNodeError("Duplicate linear spring index "+top10::util::toString(idx), child);
	m_lin_spring_specs[idx] = new LinearSpringProperties;
	m_lin_spring_specs[idx]->load(child);

	child = child->NextSiblingElement(child->Value());
      }

      // Load all angular springs
      child = node->FirstChildElement(AngularSpringProperties::NODE_NAME);
      while (child)
      {
	int idx = 0;
	child->Attribute("index", &idx);
	if (m_ang_spring_specs[idx])
	  logXmlNodeError("Duplicate angular spring index "+top10::util::toString(idx), child);
	m_ang_spring_specs[idx] = new AngularSpringProperties;
	m_ang_spring_specs[idx]->load(child);

	child = child->NextSiblingElement(child->Value());
      }

      // Load all linear spring attachments
      child = node->FirstChildElement("body_lin_attach");
      while (child)
      {
	SpringAttachment attch;
	int s = attch.loadXml(child);
	if (s)
	{
	  logXmlNodeError("Error in linear spring attachment", child, top10::util::Log::Error);
	  status = s;
	}
	else
	  m_lin_springs.push_back(attch);

	child = child->NextSiblingElement(child->Value());
      }

      // Load all angular spring attachments
      child = node->FirstChildElement("body_ang_attach");
      while (child)
      {
	SpringAttachment attch;
	int s = attch.loadXml(child);
	if (s)
	{
	  logXmlNodeError("Error in angular spring attachment", child, top10::util::Log::Error);
	  status = s;
	}
	else
	  m_ang_springs.push_back(attch);

	child = child->NextSiblingElement(child->Value());
      }

      // Load all wheel attachment points
      child = node->FirstChildElement("wheel_lin_attach");
      while (child)
      {
	SpringAttachment attch;
	int s = attch.loadXml(child);
	if (s)
	{
	  logXmlNodeError("Error in wheel attachment", child, top10::util::Log::Error);
	  status = s;
	}
	else
	  m_wheel_lin_attach.push_back(attch);
	  
	child = child->NextSiblingElement(child->Value());
      }

      child = node->FirstChildElement("wheel_ang_attach");
      while (child)
      {
	SpringAttachment attch;
	int s = attch.loadXml(child);
	if (s)
	{
	  logXmlNodeError("Error in wheel attachment", child, top10::util::Log::Error);
	  status = s;
	}
	else
	  m_wheel_ang_attach.push_back(attch);
	  
	child = child->NextSiblingElement(child->Value());
      }

      return status;
    }




    int KartProperties::saveXml(TiXmlElement* node) const
    {
      TiXmlElement child("");
      m_engine->save(&child);
      node->InsertEndChild(child);

      TiXmlElement child2("");
      m_gear_box->save(&child2);
      node->InsertEndChild(child2);

      TiXmlElement child3("");
      m_clutch->save(&child3);
      node->InsertEndChild(child3);

      TiXmlElement child4("");
      m_axle->save(&child4);
      node->InsertEndChild(child4);

      TiXmlElement child5("");
      m_front_wheels->save(&child5);
      child5.SetAttribute("location", "front");
      node->InsertEndChild(child5);

      TiXmlElement child6("");
      m_rear_wheels->save(&child6);
      child6.SetAttribute("location", "rear");
      node->InsertEndChild(child6);

      for (int i=0; i<m_bodies.size(); ++i)
      {
	TiXmlElement child("");
	if (m_bodies[i])
	{
	  m_bodies[i]->save(&child);
	  child.SetAttribute("index", i);
	  node->InsertEndChild(child);
	}
      }

      for (int i=0; i<m_lin_spring_specs.size(); ++i)
      {
	TiXmlElement child("");
	if (m_lin_spring_specs[i])
	{
	  m_lin_spring_specs[i]->save(&child);
	  child.SetAttribute("index", i);
	  node->InsertEndChild(child);
	}
      }

      for (int i=0; i<m_ang_spring_specs.size(); ++i)
      {
	TiXmlElement child("");
	if (m_ang_spring_specs[i])
	{
	  m_ang_spring_specs[i]->save(&child);
	  child.SetAttribute("index", i);
	  node->InsertEndChild(child);
	}
      }

      for (unsigned int i=0; i<m_lin_springs.size(); ++i)
      {
	TiXmlElement child("body_lin_attach");
	m_lin_springs[i].saveXml(&child);
	node->InsertEndChild(child);
      }

      for (unsigned int i=0; i<m_ang_springs.size(); ++i)
      {
	TiXmlElement child("body_ang_attach");
	m_ang_springs[i].saveXml(&child);
	node->InsertEndChild(child);
      }

      for (unsigned int i=0; i<m_wheel_lin_attach.size(); ++i)
      {
	TiXmlElement child("wheel_lin_attach");
	m_wheel_lin_attach[i].saveXml(&child);
	node->InsertEndChild(child);
      }

      for (unsigned int i=0; i<m_wheel_ang_attach.size(); ++i)
      {
	TiXmlElement child("wheel_ang_attach");
	m_wheel_ang_attach[i].saveXml(&child);
	node->InsertEndChild(child);
      }

      return 0;
    }




    void KartProperties::clearState()
    {
      m_engine->clearState();
      m_gear_box->clearState();
      m_clutch->clearState();
      m_axle->clearState();
      m_front_wheels->clearState();
      m_rear_wheels->clearState();

      m_bodies.clear();
      m_lin_springs.clear();
      m_ang_springs.clear();
      m_wheel_lin_attach.clear();
      m_wheel_ang_attach.clear();
    }




    void KartProperties::getBodies(top10::util::RefArray<const RigidBodyProperties>* bodies) const
    {
      bodies->clear();
      for (int i=0; i<m_bodies.size(); ++i)
	bodies->push_back(m_bodies[i]);
    }




    void KartProperties::getLinearAttachments(std::vector<SpringAttachment>* springs) const
    {
      *springs = m_lin_springs;
    }




    void KartProperties::getLinearSprings(top10::util::RefArray<const LinearSpringProperties>* out) const
    {
      out->clear();
      for (int i=0; i<m_lin_spring_specs.size(); ++i)
	out->push_back(m_lin_spring_specs[i]);
    }




    void KartProperties::getAngularAttachments(std::vector<SpringAttachment>* springs) const
    {
      *springs = m_ang_springs;
    }




    void KartProperties::getAngularSprings(top10::util::RefArray<const AngularSpringProperties>* out) const
    {
      out->clear();
      for (int i=0; i<m_ang_spring_specs.size(); ++i)
	out->push_back(m_ang_spring_specs[i]);
    }




    void KartProperties::getWheelLinearAttachments(std::vector<SpringAttachment>* out) const
    {
      *out = m_wheel_lin_attach;
    }




    void KartProperties::getWheelAngularAttachments(std::vector<SpringAttachment>* out) const
    {
      *out = m_wheel_ang_attach;
    }




    /*
     * SpringAttachment
     */
    int SpringAttachment::loadXml(const TiXmlElement* node)
    {
      int status = 0;

      node->Attribute("index", &m_spring_idx);
      node->Attribute("body1", &m_idx_body[0]);
      node->Attribute("body2", &m_idx_body[1]);

      const TiXmlElement* vec_node = node->FirstChildElement("vec1");
      if (vec_node)
	m_vec[0].loadXml(vec_node);
      else
	status = -1;

      vec_node = node->FirstChildElement("vec2");
      if (vec_node)
	m_vec[1].loadXml(vec_node);
      else
	status = -1;

      return status;
    }




    int SpringAttachment::saveXml(TiXmlElement* node) const
    {
      node->SetAttribute("index", m_spring_idx);
      node->SetAttribute("body1", m_idx_body[0]);
      node->SetAttribute("body2", m_idx_body[1]);
      TiXmlElement vec1("vec1");
      m_vec[0].saveXml(&vec1);
      TiXmlElement vec2("vec2");
      m_vec[1].saveXml(&vec2);
      node->InsertEndChild(vec1);
      node->InsertEndChild(vec2);
      
      return 0;
    }

  }
}
