/*
  Top 10, a racing simulator
  Copyright (C) 2003-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#ifndef TOP10_INTEGRATOR_HH
#define TOP10_INTEGRATOR_HH

#include "math/Vertex.hh" // Used only by the helpers.
#include "math/Matrix.hh"

namespace top10
{
  namespace physX
  {
    /*
      From http://www.gaffer.org/game-physics/integration-basics/
      */
    /*!
      State must have get/set methods for Pos and Speed + a default constructor.
      Derivative must have get/set methods for Speed and Accel + a default and a copy constructor.
      Real is double or float.
      Acceleration is a function taking a State and a Real, returning a type assignable to Derivative::accel.
      */
    template <typename State, typename Derivative, typename Real, typename Acceleration>
    Derivative
      evaluate(const State& initial, Real t, Real dt, const Derivative& d, const Acceleration& acceleration)
    {
      State state;
      state.setPos(initial.getPos() + d.getSpeed()*dt);
      state.setSpeed(initial.getSpeed() + d.getAccel()*dt);

      Derivative output;
      output.setSpeed(state.getSpeed());
      output.setAccel(acceleration(state, t+dt));

      return output;
    }

    template <typename State, typename Derivative, typename Real, typename Acceleration>
    Derivative
      evaluate(const State& initial, Real t, const Acceleration& acceleration)
    {
      Derivative output;
      output.setSpeed(initial.getSpeed());
      output.setAccel(acceleration(initial, t));

      return output;
    }

    template <typename State, typename Derivative, typename Real, typename Acceleration>
    void
      integrate(State &state, Real t, Real dt, const Acceleration& accel)
    {
      Derivative a = evaluate<State, Derivative, Real, Acceleration>(state, t, accel);
      Derivative b = evaluate<State, Derivative, Real, Acceleration>(state, t, dt*0.5f, a, accel);
      Derivative c = evaluate<State, Derivative, Real, Acceleration>(state, t, dt*0.5f, b, accel);
      Derivative d = evaluate<State, Derivative, Real, Acceleration>(state, t, dt, c, accel);

      const typename State::SpeedT      dxdt = 1.0f/6.0f * (a.getSpeed() + 2.0f*(b.getSpeed() + c.getSpeed()) + d.getSpeed());
      const typename Derivative::AccelT dvdt = 1.0f/6.0f * (a.getAccel() + 2.0f*(b.getAccel() + c.getAccel()) + d.getAccel());

      state.setPos(state.pos + dxdt * dt);
      state.setSpeed(state.speed + dvdt * dt);
    }


    /*
     * Integration for angular movements represented using rotation matrices.
     */

    struct AngularState
    {
      top10::math::Matrix3 orient;
      top10::math::Vector w_speed;
    };

    struct AngularDerivative
    {
      top10::math::Vector w_speed;
      top10::math::Vector w_accel;
    };

    template <typename Real, typename Acceleration>
    AngularDerivative evaluateAngular(const AngularState& initial, Real t, Real dt, const AngularDerivative& d, const Acceleration& acceleration)
    {
      AngularState state;
      state.orient = initial.orient;
      updateOrient(state.orient, dt*d.w_speed);
      state.w_speed = initial.w_speed + dt*d.w_accel;

      AngularDerivative output;
      output.w_speed = state.w_speed;
      output.w_accel = acceleration(state, t+dt);

      return output;
    }

    template <typename Real, typename Acceleration>
    AngularDerivative
      evaluateAngular(const AngularState& initial, Real t, const Acceleration& acceleration)
    {
      AngularDerivative output;
      output.w_speed = initial.w_speed;
      output.w_accel = acceleration(initial, t);

      return output;
    }

    /*!
    \param orient Orientation at timestep t,
    \param w_speed Angular speed at timestep t, in the frame of orient.
    */
    template <typename Real, typename Accelerator>
    void
      integrateAngular(top10::math::Matrix3& orient, top10::math::Vector& w_speed, Real t, Real dt, Accelerator accel)
    {
      AngularState state;
      state.orient = orient;
      state.w_speed = w_speed;

      AngularDerivative a = evaluateAngular(state, t, accel);
      AngularDerivative b = evaluateAngular(state, t, dt*0.5, a, accel);
      AngularDerivative c = evaluateAngular(state, t, dt*0.5, b, accel);
      AngularDerivative d = evaluateAngular(state, t, dt, c, accel);

      top10::math::Vector dxdt = 1.0/6.0 * (a.w_speed + 2.0 * (b.w_speed + c.w_speed) + d.w_speed);
      top10::math::Vector dvdt = 1.0/6.0 * (a.w_accel + 2.0 * (b.w_accel + c.w_accel) + d.w_accel);

      top10::math::Vector ws = dt*dxdt;
      updateOrient(orient, ws);

      w_speed = w_speed + dt*dvdt;
    }

    struct ConstantAngularAccel
    {
      ConstantAngularAccel(top10::math::Vector accel)
	: m_constant(accel)
      {
      }

      top10::math::Vector operator()(const AngularState&, double) const
      {
	return m_constant;
      }

    private:
      top10::math::Vector m_constant;
    };

    /*
     * A few helpers. Use them if the templates above give you headaches.
     */

    // A state based on T.
    template< typename T >
    struct State
    {
      typedef T PositionT;
      typedef T SpeedT;

      PositionT pos;
      SpeedT speed;

      inline PositionT getPos()   const { return pos; }
      inline SpeedT    getSpeed() const { return speed; }

      inline void setPos  (PositionT d) { pos = d; }
      inline void setSpeed(SpeedT d)    { speed = d; }
    };

    // A derivative based on T.
    template< typename T >
    struct Derivative
    {
      typedef T SpeedT;
      typedef T AccelT;

      SpeedT speed;
      AccelT accel;

      inline AccelT getAccel() const { return accel; }
      inline SpeedT getSpeed() const { return speed; }

      inline void setAccel(AccelT d) { accel = d; }
      inline void setSpeed(SpeedT d) { speed = d; }
    };

    //! Helper for common integration tasks using Vectors.
    /*!
      The caller need not explicitly specify the template types.
      */
    template<typename Component, typename Real, typename Acceleration>
    void integrateDirect(Component& pos, Component& speed, Real t, Real dt, const Acceleration& accel)
    {
      State<Component> state;
      state.setPos(pos);
      state.setSpeed(speed);

      integrate< State<Component>, Derivative<Component>, Real, Acceleration>
	(state, t, dt, accel);

      pos = state.getPos();
      speed = state.getSpeed();
    }

    // State using a Vector.
    typedef State     <top10::math::Vector> VectorState;
    // Derivative using a Vector.
    typedef Derivative<top10::math::Vector> VectorDerivative;
    // State using a double.
    typedef State     <double> DoubleState;
    // Derivative using a double.
    typedef Derivative<double> DoubleDerivative;

    struct ConstantVectorAccel
    {
      ConstantVectorAccel(top10::math::Vector accel)
	: m_constant(accel)
      {
      }

      top10::math::Vector operator()(const VectorState&, double) const
      {
	return m_constant;
      }

    private:
      top10::math::Vector m_constant;
    };

    struct ConstantDoubleAccel
    {
      ConstantDoubleAccel(double accel)
	: m_constant(accel)
      {
      }

      double operator()(const DoubleState&, double) const
      {
	return m_constant;
      }

    private:
      double m_constant;
    };

  }
}

#endif
