/*
  Top 10, a racing simulator
  Copyright (C) 2003-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "GearBoxProperties.hh"

namespace top10
{
  namespace physX
  {
    const char* GearBoxProperties::NODE_NAME = "gearbox_properties";

    GearBoxProperties::GearBoxProperties()
      : top10::util::XmlDumpable(NODE_NAME),
      m_backward_ratios(),
      m_forward_ratios()
    {
    }

    void GearBoxProperties::insertRatio(double r)
    {
      if (r > 0.0)
      {
	std::vector<double>::iterator it = m_forward_ratios.begin();
	while (it != m_forward_ratios.end() && *it < r)
	  ++it;
	m_forward_ratios.insert(it, r);
      }
      else
      {
	std::vector<double>::iterator it = m_backward_ratios.begin();
	while (it != m_backward_ratios.end() && *it > r)
	  ++it;
	m_backward_ratios.insert(it, r);
      }
    }

    double GearBoxProperties::getRatio(int gear) const
    {
      if (gear == 0)
	return 0.0;

      else if (gear > 0)
	return m_forward_ratios.at(gear -1);

      else
	return m_backward_ratios.at(-gear -1);
    }

    int GearBoxProperties::getNumForwardRatios() const
    {
      return m_forward_ratios.size();
    }

    int GearBoxProperties::getNumBackwardRatios() const
    {
      return m_backward_ratios.size();
    }

    bool GearBoxProperties::isValid(int gear) const
    {
      if (gear == 0)
	return true;

      if (gear > 0 && (unsigned int)gear <= m_forward_ratios.size())
	return true;

      else if (gear < 0 && (unsigned int)-gear <= m_backward_ratios.size())
	return true;

      return false;
    }

    void GearBoxProperties::clearState()
    {
      m_backward_ratios.clear();
      m_forward_ratios.clear();
    }

    int GearBoxProperties::loadXml(const TiXmlElement* node)
    {
      int status = 0;
      int status2 = 0;

      const TiXmlElement* child = node->FirstChildElement("ratio");
      while (child)
      {
	double val;
	status2 = child->QueryDoubleAttribute("value", &val);
	if (status2)
	{
	  status = status2;
	  logXmlNodeError("Could not get value from node ratio", child, top10::util::Log::Error);
	}
	else
	  insertRatio(val);

	child = child->NextSiblingElement("ratio");
      }

      return status;
    }

    int GearBoxProperties::saveXml(TiXmlElement* node) const
    {
      int status = 0;

      for (std::vector<double>::const_iterator it = m_backward_ratios.begin();
	it != m_backward_ratios.end();
	++it)
      {
	TiXmlElement item("ratio");
	item.SetDoubleAttribute("value", *it);
	node->InsertEndChild(item);
      }

      
      for (std::vector<double>::const_iterator it = m_forward_ratios.begin();
	it != m_forward_ratios.end();
	++it)
      {
	TiXmlElement item("ratio");
	item.SetDoubleAttribute("value", *it);
	node->InsertEndChild(item);
      }

      return status;
    }
  }
}
