/*
  Top 10, a racing simulator
  Copyright (C) 2003-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "EngineProperties.hh"

namespace top10
{
  namespace physX
  {
    const char* EngineProperties::NODE_NAME = "engine_properties";

    EngineProperties::EngineProperties()
      : top10::util::XmlDumpable(NODE_NAME),
      m_curve_throttle(),
      m_curve_idle(),
      m_inertia(0.0)
    {
    }

    void EngineProperties::addPointThrottle(double rad_per_sec, double torque)
    {
      m_curve_throttle.addXY(rad_per_sec, torque);
    }

    void EngineProperties::addPointIdle(double rad_per_sec, double torque)
    {
      m_curve_idle.addXY(rad_per_sec, torque);
    }

    double EngineProperties::getTorque(double throttle, double rad_per_sec) const
    {
      assert(throttle >= 0 && throttle <= 1.0);

      return m_curve_idle.getY(rad_per_sec) * (1.0 - throttle) +
	m_curve_throttle.getY(rad_per_sec) * throttle;
    }

    double EngineProperties::getInertia() const
    {
      return m_inertia;
    }

    void EngineProperties::clearState()
    {
      m_curve_throttle = top10::math::Curve2D();
      m_curve_idle = top10::math::Curve2D();
      m_inertia = 0.0;
    }

    int EngineProperties::loadXml(const TiXmlElement* node)
    {
      int status = 0;
      int status2 = 0;
      const TiXmlElement* child;
      
      child = node->FirstChildElement("curve_throttle");
      if (child != 0)
      {
	status = m_curve_throttle.loadXml(child);
      }
      else
      {
	status = -1;
	logXmlNodeError("Missing curve_throttle node", node, top10::util::Log::Error);
      }

      child = node->FirstChildElement("curve_idle");
      if (child != 0)
      {
	status2 = m_curve_idle.loadXml(child);
	if (status2 != 0)
	  status = status2;
      }
      else
      {
	status = -1;
	logXmlNodeError("Missing curve_idle node", node, top10::util::Log::Error);
      }

      status2 = node->QueryDoubleAttribute("inertia", &m_inertia);
      if (status2 != 0)
	status = status2;

      return status;
    }

    int EngineProperties::saveXml(TiXmlElement* node) const
    {
      int status = 0;
      
      node->SetDoubleAttribute("inertia", m_inertia);

      TiXmlElement child("curve_throttle");
      m_curve_throttle.saveXml(&child);
      node->InsertEndChild(child);

      TiXmlElement child2("curve_idle");
      m_curve_idle.saveXml(&child2);
      node->InsertEndChild(child2);

      return status;
    }
  }
}
