/*
  Top 10, a racing simulator
  Copyright (C) 2003-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "AngularSpringProperties.hh"

namespace top10
{
  namespace physX
  {
    const char* AngularSpringProperties::NODE_NAME = "rot_spring_properties";

    AngularSpringProperties::AngularSpringProperties()
      : top10::util::XmlDumpable(NODE_NAME),
      m_stiffness(0.0),
      m_damping(0.0)
    {
    }

    void AngularSpringProperties::setStiffness(double v)
    {
      m_stiffness = v;
    }

    void AngularSpringProperties::setDamping(double v)
    {
      m_damping = v;
    }

    top10::math::Vector
      AngularSpringProperties::getTorque(top10::math::Vector ref,
                                         top10::math::Vector dir,
				         top10::math::Vector w_speed) const
    {
      top10::math::Vector ret;
      top10::math::Vector normal = dir^ref;

      ret = (m_stiffness + m_damping*(normal*w_speed))*normal;

      return ret;
    }

    int AngularSpringProperties::loadXml(const TiXmlElement* node)
    {
      int status = 0;

      if (node->QueryDoubleAttribute("stiff", &m_stiffness))
      {
	logXmlNodeError("Missing attribute 'stiff'", node, top10::util::Log::Error);
	status = -1;
      }

      if (node->QueryDoubleAttribute("damp", &m_damping))
      {
	logXmlNodeError("Missing attribute 'damp'", node, top10::util::Log::Error);
	status = -1;
      }

      return status;
    }

    int AngularSpringProperties::saveXml(TiXmlElement* node) const
    {
      node->SetDoubleAttribute("stiff", m_stiffness);
      node->SetDoubleAttribute("damp", m_damping);
      return 0;
    }

    void AngularSpringProperties::clearState()
    {
      m_stiffness = 0.0;
      m_damping = 0.0;
    }

  }
}
