/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/
#include "intersections.hh"

using namespace top10::math;

// Defined in extras/tribox.c
extern "C" int planeBoxOverlap(double normal[3], double d, double maxbox[3]);
extern "C" int triBoxOverlap(double boxcenter[3],double boxhalfsize[3], double triverts[3][3]);

bool top10::math::intersect(AxisAlignedBox b1, AxisAlignedBox b2)
{
  Vector d = b1.getCenter()-b2.getCenter();
  double s1[3];
  double s2[3];
  b1.getEdgeSizes(s1);
  b2.getEdgeSizes(s2);

  for (int i=0; i<3; ++i) {
    if (fabs(d[i]) < 0.5*(s1[i] + s2[i])) return true;
  }
  return false;
}

//FIXME: this algorithm is not correct :(
bool top10::math::intersect(AxisAlignedBox aabox, Box box, bool try_other)
{
  // Check if box is one side of one of aabox's planes
  double sizes[3];
  aabox.getEdgeSizes(sizes);
  int planes = 63;   // Bitfield keeping track of the planes separating the two boxes
  Vector center = aabox.getCenter();
  double x1 = center.x - sizes[0]/2.0;
  double x2 = center.x + sizes[0]/2.0;
  double y1 = center.y - sizes[1]/2.0;
  double y2 = center.y + sizes[1]/2.0;
  double z1 = center.z - sizes[2]/2.0;
  double z2 = center.z + sizes[2]/2.0;

  Vector vertices[8];
  box.getVertices(vertices);

  for (int i=0; planes && i<8; ++i) {
    const Vector& v(vertices[i]);
    bool inside = true;
    if (v.x >= x1) planes &= ~1; else inside = false;
    if (v.x <= x2) planes &= ~2; else inside = false;
    if (v.y >= y1) planes &= ~4; else inside = false;
    if (v.y <= y2) planes &= ~8; else inside = false;
    if (v.z >= z1) planes &= ~16; else inside = false;
    if (v.z <= z2) planes &= ~32; else inside = false;

    // v is inside aabox
    if (inside) return true;
  }
  // There is a plane separating the two boxes, hence they do not intersect
  if (planes) return false;
  // We tried all possible planes, could not find one, hence boxes intersect
  if (!try_other) return true;
  
  // Try once more when swaping boxes.
  Matrix3 M = box.getOrient().transpose();
  Box box2(aabox.getCenter(), M*Vector(sizes[0]/2.0,0,0), M*Vector(0,sizes[1]/2.0,0), M*Vector(0,0,sizes[2]/2.0));
  box.getEdgeSizes(sizes);
  AxisAlignedBox aabox2(box.getCenter(), sizes[0]/2.0, sizes[1]/2.0, sizes[2]/2.0);
  return intersect(aabox2, box2, false);
}

bool top10::math::intersect(Plane plane, AxisAlignedBox box)
{
  double normal[3];
  const Vector& N(plane.normal);
  normal[0] = N.x;
  normal[1] = N.y;
  normal[2] = N.z;

  double maxbox[3];
  double sizes[3];
  box.getEdgeSizes(sizes);
  for (int i=0; i<3; ++i) maxbox[i] = sizes[i]/2.0;

  return planeBoxOverlap(normal, plane.d, maxbox) != 0;
}

bool top10::math::intersect(Triangle triangle, AxisAlignedBox box)
{
  double triverts[3][3];
  for (int i=0; i<3; ++i) {
    for (int j=0; j<3; ++j) {
      triverts[i][j] = triangle.p[i][j];
    }
  }

  double boxcenter[3];
  for (int i=0; i<3; ++i) {
    boxcenter[i] = box.getCenter()[i];
  }

  double boxhalfsize[3];
  double sizes[3];
  box.getEdgeSizes(sizes);
  for (int i=0; i<3; ++i) {
    boxhalfsize[i] = sizes[i]/2.0;
  }

  return triBoxOverlap(boxcenter, boxhalfsize, triverts) != 0;
}

bool top10::math::intersect(Plane plane, Box box)
{
  Matrix3 M = box.getOrient().transpose();
  plane.translate(-box.getCenter());
  plane.rotate(M);
  double normal[3];
  for (int i=0; i<3; ++i) normal[i] = plane.normal[i];

  double maxbox[3];
  double sizes[3];
  box.getEdgeSizes(sizes);
  for (int i=0; i<3; ++i) maxbox[i] = sizes[i]/2.0;

  return planeBoxOverlap(normal, plane.d, maxbox) != 0;
}

bool top10::math::intersect(Triangle triangle, Box box)
{
  Matrix3 M = box.getOrient().transpose();
  triangle.translate(-box.getCenter());
  triangle.rotate(M);

  double triverts[3][3];
  for (int i=0; i<3; ++i) {
    for (int j=0; j<3; ++j) {
      triverts[i][j] = triangle.p[i][j];
    }
  }

  double boxcenter[3];
  for (int i=0; i<3; ++i) boxcenter[i] = 0.0;

  double boxhalfsize[3];
  double sizes[3];
  box.getEdgeSizes(sizes);
  for (int i=0; i<3; ++i) {
    boxhalfsize[i] = sizes[i]/2.0;
  }

  return triBoxOverlap(boxcenter, boxhalfsize, triverts) != 0;
}

bool top10::math::intersect(Mesh mesh, AxisAlignedBox box)
{
  const std::vector<Mesh::Face>* faces = mesh.getFaces();
  const std::vector<Vector>* vertices = mesh.getVertices();
  
  //TODO: use an octree to partition faces of the mesh
  for (std::vector<Mesh::Face>::const_iterator face_it = faces->begin(); face_it != faces->end(); ++face_it) {
    Triangle t(vertices->at(face_it->idxs[0]), vertices->at(face_it->idxs[1]), vertices->at(face_it->idxs[2]));
    if (intersect(t, box)) return true;
  }
  return false;
}

bool top10::math::intersect(Mesh mesh, Box box)
{
  const std::vector<Mesh::Face>* faces = mesh.getFaces();
  const std::vector<Vector>* vertices = mesh.getVertices();
  
  //TODO: use an octree to partition faces of the mesh
  for (std::vector<Mesh::Face>::const_iterator face_it = faces->begin(); face_it != faces->end(); ++face_it) {
    Triangle t(vertices->at(face_it->idxs[0]), vertices->at(face_it->idxs[1]), vertices->at(face_it->idxs[2]));
    if (intersect(t, box)) return true;
  }
  return false;
}

bool top10::math::intersect(Plane p, Ray r)
{
  if (r.isFinite()) {
    Vector res;
    return intersect(p, r.getOrigin(), r.getOrigin() + r.getLength()*r.getDirection(), res);
  }
  else {
    Vector res;
    double d;
    return intersectRay(p, r.getOrigin(), r.getDirection(), res, d);
  }  
}

bool top10::math::intersect(Triangle t, Ray r)
{
  Vector v;
  double d;
  bool res = t.intersectRay(r.getOrigin(), r.getDirection(), v, d);
  if (!res) return false;
  if (d < 0.0) return false;
  if (r.isFinite() && d > r.getLength()) return false;
  return true;
}

bool top10::math::intersect(Mesh m, Ray r, double* dist)
{
  const std::vector<Vector>* vertices = m.getVertices();
  const std::vector<Mesh::Face>* faces = m.getFaces();
  bool found = false;
  for (std::vector<Mesh::Face>::const_iterator face_it = faces->begin(); face_it != faces->end(); ++face_it) {
    Triangle tri(vertices->at(face_it->idxs[0]), vertices->at(face_it->idxs[1]), vertices->at(face_it->idxs[2]));
    Vector v;
    double d;
    if (tri.intersectRay(r.getOrigin(), r.getDirection(), v, d)) {
      if (dist == 0) return true;
      if (*dist > d && d >= 0.0) *dist = d;
      found = true;
    }
  } 
  return found;
}
