/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_MATH_VEC2D_HH
#define TOP10_MATH_VEC2D_HH

#include <iostream>

#ifndef SMALL_VECTOR
#define SMALL_VECTOR (1e-4)
#endif

namespace top10 {
  namespace math {

    //! Represents a 2d vector
    class Vec2D {
    public:
      double x;
      double y;

    public:
      Vec2D(double _x=0, double _y=0): x(_x), y(_y) {};

      Vec2D& operator+=(const Vec2D&);
      Vec2D& operator-=(const Vec2D&);

      Vec2D& operator/=(double);
      Vec2D& operator*=(double);

      //! Returns the opposite vector
      Vec2D operator- ();

      //! square size
      double size2() const;

      double size() const;

      inline Vec2D rotate90() const {return Vec2D(-y, x);}
    };
  };
};

//! Dot product.  
double operator*(const top10::math::Vec2D&, const top10::math::Vec2D&);

//! Cross product.  
double operator^(const top10::math::Vec2D&, const top10::math::Vec2D&);

//! Addition of two vectors
inline top10::math::Vec2D operator+ (const top10::math::Vec2D& a,
				     const top10::math::Vec2D& b)
{
  top10::math::Vec2D ret = a;
  return ret+=b;
}

//! Subtraction of two vectors
inline top10::math::Vec2D operator- (const top10::math::Vec2D& a,
				     const top10::math::Vec2D& b)
{
  top10::math::Vec2D ret = a;
  return ret-=b;
}

//! Multiplication by a scalar
top10::math::Vec2D operator*(double, top10::math::Vec2D);
    
//! Multiplication by a scalar
inline top10::math::Vec2D operator*(const top10::math::Vec2D& a, double b)
{
  return b*a;
}

//! Division by a scalar
top10::math::Vec2D operator/(top10::math::Vec2D, double);

//! Human readable output
inline std::ostream& operator<<(std::ostream& out, const top10::math::Vec2D& v)
{
  return out<<v.x<<" "<<v.y;
}

bool operator==(const top10::math::Vec2D&, const top10::math::Vec2D&);
inline bool operator!=(const top10::math::Vec2D& A, const top10::math::Vec2D& B) { return !(A==B); }

//! Check intersection of two 2d segments
bool segment_intersection(const top10::math::Vec2D& pt1,
			  const top10::math::Vec2D& dir1,
			  const top10::math::Vec2D& pt2,
			  const top10::math::Vec2D& dir2);

//! Compute intersection of two 2d segments
/*! Puts in t a number between 0 and 1 such that the intersection point is pt1 + r*dir1 */
bool segment_intersection(const top10::math::Vec2D& pt1,
			  const top10::math::Vec2D& dir1,
			  const top10::math::Vec2D& pt2,
			  const top10::math::Vec2D& dir2,
			  double& t);
#endif
