/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_MATHTRIANGULATION_HH
#define TOP10_MATHTRIANGULATION_HH

#include <map>
#include <set>
#include <vector>
#include <list>

#include <CGAL/Exact_predicates_inexact_constructions_kernel.h>
#include <CGAL/Constrained_Delaunay_triangulation_2.h>

#include "Triangle.hh"
#include "Mesh.hh"
#include "Vec2D.hh"
#include "util/error.hh"
#include "util/strconv.hh"
#include "util/Log.hh"
#include "util/PairLexOrder.hh"

namespace top10 {
namespace math {

/*!
  \param VectorT type of vertices
  \param VectorTAccessor type of class allowing access to components of VectorT.
    Must define methods double getX(const VectorT&) and double getY(const VectorT&)
    and double getZ(const VectorT&)
  \param VectorTMerger type of class used to merge two different VectorT with identical positions.
    Must define void merge(VectorT& dest, const VectorT& add)
    The resulting of the merge must not affect the ordering computed by VertexTLexOrder
  \param VectorTLexOrder ordering used for mappings of VectorT
*/
template< typename VectorT, typename VectorTAccessor, typename VectorTMerger, typename VectorTLexOrder >
class Triangulation
{
private:
  struct K : CGAL::Exact_predicates_inexact_constructions_kernel {};
  typedef typename CGAL::Triangulation_vertex_base_2<K>                     Vb;
  typedef typename CGAL::Constrained_triangulation_face_base_2<K>           Fb;
  typedef typename CGAL::Triangulation_data_structure_2<Vb,Fb>              TDS;
  typedef typename CGAL::No_intersection_tag                                Itag;
  typedef typename CGAL::Constrained_Delaunay_triangulation_2<K, TDS, Itag> CDT;
  typedef typename CDT::Point                                               Point;

public:

  class ConflictingConstraint: public top10::util::Error {
  public:
    ConflictingConstraint(const VectorT& c1v1, const VectorT& c1v2, const VectorT& c2v1, const VectorT& c2v2);

    const VectorT &c1v1;
    const VectorT &c1v2;
    const VectorT &c2v1;
    const VectorT &c2v2;
  };

  class UnknownVertex: public top10::util::Error {
  public:
    UnknownVertex(const Point&);
  };
      
  struct Triangle {
    unsigned int p[3];
  };
  
  struct Mesh {
    std::vector<VectorT> vertices;
    std::vector<Triangle> faces;
  };
  
public:
  //! Triangulate an area included in a rectangle.
  /*! All vertices and constraints added later must fall within this region. */
  Triangulation(double x_min, double z_min, double x_max, double z_max, const VectorTAccessor&);
  
  void addVertex(const VectorT&);
  bool checkConstraint(const VectorT&, const VectorT&, VectorT& c2v1, VectorT& c2v2);
  void addConstraint(const VectorT&, const VectorT&);
  void addConstraints(const std::vector<VectorT>&);
  
  Mesh getTriangles(bool remove_holes=false);
  bool getTriangle(const VectorT&, VectorT&, VectorT&, VectorT&) const;

  static std::string toString(const VectorT&, VectorTAccessor&);
  
  ~Triangulation();

private:
  enum State { Creating, Using };
  typedef std::map< VectorT, Point, VectorTLexOrder > VectorToGts;
  typedef std::map< Point, VectorT > GtsToVector;
  typedef std::set< std::pair< VectorT, VectorT >, top10::util::PairLexOrder< VectorT, VectorTLexOrder> > Constraints;
  
  //! Allows to access components of vertices
  VectorTAccessor acc;
  //! Keeps track of whether we are initialising or querying this triangulation
  State state;
  //! The CGAL object representing the triangulation
  CDT cdt;
  //! Mapping from top10's vertices to Gts'
  VectorToGts to_gts;
  //! Mapping from Gts' vertices to top10's
  GtsToVector from_gts;
  //! Keep track of the constraints
  Constraints constraints;
  //! Used to generate meshes
  std::map<Point, unsigned int> gts2id;
  //! Result of the triangulation
  Mesh mesh;
  
  Point toGts(const VectorT&) const;
  
  void createMesh();
  void addToMesh(const typename CDT::Face&);
  unsigned int getVertexId(Point v);
  
  //! Forbidden
  Triangulation<VectorT, VectorTAccessor, VectorTMerger, VectorTLexOrder>& operator=(const Triangulation<VectorT, VectorTAccessor, VectorTMerger, VectorTLexOrder>&);
  //! Forbidden
  Triangulation<VectorT, VectorTAccessor, VectorTMerger, VectorTLexOrder>(const Triangulation<VectorT, VectorTAccessor, VectorTMerger, VectorTLexOrder>&);
};

//#include "Triangulation-template.cpp"

}
}

#endif
