/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

template<typename Vt, typename At, typename Mt, typename Ot>
Triangulation<Vt, At, Mt, Ot>::Triangulation(double x_min, double z_min, double x_max, double z_max, const At& acc):
    acc(acc),
    state(Creating)
{
}

template<typename Vt, typename At, typename Mt, typename Ot>
Triangulation<Vt, At, Mt, Ot>::~Triangulation()
{
}

template<typename Vt, typename At, typename Mt, typename Ot>
void Triangulation<Vt, At, Mt, Ot>::addVertex(const Vt& v)
{
  assert(state == Creating);
  assert(to_gts.size() == from_gts.size());
  
  typename Triangulation<Vt, At, Mt, Ot>::VectorToGts::iterator find_it = to_gts.find(v);

  // The vertex does not exist yet
  if (find_it == to_gts.end())
  {
    Point gts_v = toGts(v);
    
    to_gts[v] = gts_v;
    assert(from_gts.find(gts_v) == from_gts.end());
    from_gts[gts_v] = v;

    cdt.insert(gts_v);
  }
  // A vertex with the same coordinates exists, merge their properties
  else if (!(v == find_it->first))
  {
    Mt merger;

    // Update the vertex in to_gts
    //  Get rid of the constness of the key. That's ok because we require that merging properties does not affect ordering.
    const Vt& found(find_it->first);
    Vt* found_rw = const_cast<Vt*>(&found);

#ifndef NDEBUG
    Ot lex_order;
    //  Check the requirement on merging not affecting ordering.
    assert((!lex_order(found, v)) && (!lex_order(v, found)));
#endif

    merger.merge(*found_rw, v);

    // Update the vertex in from_gts
    typename Triangulation<Vt, At, Mt, Ot>::GtsToVector::iterator dnif_it = from_gts.find(find_it->second);
    assert(dnif_it != from_gts.end());
    merger.merge(dnif_it->second, v);
  }
}
template<typename Vt, typename At, typename Mt, typename Ot>
bool Triangulation<Vt, At, Mt, Ot>::checkConstraint(const Vt& v1, const Vt& v2, Vt &c2v1, Vt &c2v2)
{
  Vec2D vec11(acc.getZ(v1), acc.getX(v1));
  Vec2D vec12(acc.getZ(v2), acc.getX(v2));
  Vec2D dir1 = vec12;
  dir1 -= vec11;

  for (typename Constraints::const_iterator it = constraints.begin(); it != constraints.end(); ++it)
  {
    Vec2D vec21(acc.getZ(it->first), acc.getX(it->first));
    Vec2D vec22(acc.getZ(it->second), acc.getX(it->second));
    Vec2D dir2 = vec22;
    dir2 -= vec21;

    if (vec11 != vec21 && vec11 != vec22 && vec12 != vec21 && vec12 != vec22 && segment_intersection(vec11, dir1, vec21, dir2))
    {
      c2v1 = it->first;
      c2v2 = it->second;
      return true;
    }
  }

  return false;
}

template<typename Vt, typename At, typename Mt, typename Ot>
void Triangulation<Vt, At, Mt, Ot>::addConstraint(const Vt& v1, const Vt& v2)
{
  assert(state == Creating);
  
  typename VectorToGts::const_iterator f1, f2;
  
  addVertex(v1);
  f1 = to_gts.find(v1);
  assert(f1 != to_gts.end());
  
  addVertex(v2);
  f2 = to_gts.find(v2);
  assert(f2 != to_gts.end());
  
  if (f1 == f2) {
    top10::util::Log::getSingle()->send(top10::util::Log::Warning, "Triangulation<>::addConstraint",
    "Degenerate constraint" + toString(v1, acc) + " -> " + toString(v2, acc));
    return;
  }
  
  if (constraints.find(std::make_pair(v1, v2)) != constraints.end())
    return;
  
  if (constraints.find(std::make_pair(v2, v1)) != constraints.end())
    return;
  
  // Check if there exists a constraint intersecting the new one
  Vt c2v1, c2v2;
  if (checkConstraint(v1, v2, c2v1, c2v2))
    throw ConflictingConstraint(v1,v2, c2v1, c2v2);

  constraints.insert(std::make_pair(v1,v2));
  cdt.insert_constraint(f1->second, f2->second);
}

template<typename Vt, typename At, typename Mt, typename Ot>
    void Triangulation<Vt, At, Mt, Ot>::addConstraints(const std::vector<Vt>& constraints)
{
  if (constraints.size() < 2) return;
    
  typename std::vector<Vt>::const_iterator it1, it2;
  it1=constraints.begin();
  it2=it1;
  ++it2;
  for (; it2 != constraints.end(); ++it1, ++it2)
  {
    addConstraint(*it1, *it2);
  }
  addConstraint(constraints.back(), constraints.front());
}

template<typename Vt, typename At, typename Mt, typename Ot>
typename Triangulation<Vt, At, Mt, Ot>::Mesh
Triangulation<Vt, At, Mt, Ot>::getTriangles(bool remove_holes)
{
  if (state == Creating) createMesh();
  state = Using;
  return mesh;
}

template<typename Vt, typename At, typename Mt, typename Ot>
bool Triangulation<Vt, At, Mt, Ot>::getTriangle(const Vt& pt, Vt& tri0, Vt& tri1, Vt& tri2) const
{
  const Point query_pt = toGts(pt);
  const typename CDT::Face_handle handle = cdt.locate( query_pt );
  if (handle != typename CDT::Face_handle() /* null handle */)
  {
    const Point pt0 = handle->vertex(0)->point();
    const Point pt1 = handle->vertex(1)->point();
    const Point pt2 = handle->vertex(2)->point();

    typename GtsToVector::const_iterator find_it0 = from_gts.find(pt0);
    typename GtsToVector::const_iterator find_it1 = from_gts.find(pt1);
    typename GtsToVector::const_iterator find_it2 = from_gts.find(pt2);

    if (find_it0 != from_gts.end() && find_it1 != from_gts.end() && find_it2 != from_gts.end())
    {
      tri0 = find_it0->second;
      tri1 = find_it1->second;
      tri2 = find_it2->second;

      return true;
    }
    else
      return false;
  }
  return false;
}

template<typename Vt, typename At, typename Mt, typename Ot>
void Triangulation<Vt, At, Mt, Ot>::createMesh()
{
  
  // Build the mesh
  typename CDT::Finite_faces_iterator face_it, face_end;
  face_it = cdt.finite_faces_begin();
  face_end = cdt.finite_faces_end();
  for (; face_it != face_end; ++face_it)
    addToMesh(*face_it);

  // Stats
  top10::util::Log::getSingle()->send(top10::util::Log::Debug, "Triangulation<>::createMesh", "Created "+top10::util::toString(mesh.faces.size())
      +" triangles.");
  
  // Dispose of the now useless mappings
  to_gts.clear();
  from_gts.clear();
  gts2id.clear();
  constraints.clear();
}

template<typename Vt, typename At, typename Mt, typename Ot>
void Triangulation<Vt, At, Mt, Ot>::addToMesh(const typename CDT::Face& tri)
{
    // Get the vertices of the triangles
  Point v1, v2, v3;
  
  // Add the vertices to the mesh if necessary, otherwise get their position in the mesh's list of vertices
  v1 = tri.vertex(0)->point();
  v2 = tri.vertex(1)->point();
  v3 = tri.vertex(2)->point();

  // Add the new face
  try {
    Triangle new_tri;
    new_tri.p[0] = getVertexId(v1);
    new_tri.p[1] = getVertexId(v2);
    new_tri.p[2] = getVertexId(v3);
    mesh.faces.push_back(new_tri);
  }
  catch(top10::util::Error& e) {
    top10::util::Log::getSingle()->send(top10::util::Log::Warning, "Triangulation< >::gather_mesh", e);
  }
  catch(...) {
    top10::util::Log::getSingle()->send(top10::util::Log::Critical, "Triangulation< >::gather_mesh", "Unknown error");
  }
}

template<typename Vt, typename At, typename Mt, typename Ot>
unsigned int Triangulation<Vt, At, Mt, Ot>::getVertexId(Point v)
{
  typename std::map<Point, unsigned int>::const_iterator find_it = gts2id.find(v);
  unsigned int ret;
  if (find_it == gts2id.end()) {
    // Look for the VectorT corresponding to v
    typename GtsToVector::const_iterator vec_it = from_gts.find(v);
    if (vec_it == from_gts.end())
      throw typename Triangulation<Vt, At, Mt, Ot>::UnknownVertex(v);
    
    // Add it to the mesh
    ret = gts2id.size();
    gts2id[v] = ret;
    mesh.vertices.push_back(vec_it->second);
    assert(mesh.vertices.size() == ret+1);
  }
  else {
    ret = find_it->second;
  }
  
  return ret;
}

template<typename Vt, typename At, typename Mt, typename Ot>
typename Triangulation<Vt, At, Mt, Ot>::Point Triangulation<Vt, At, Mt, Ot>::toGts(const Vt& v) const
{
  assert(to_gts.find(v) == to_gts.end());
  return Point(acc.getZ(v), acc.getX(v));
}


template< typename Vt, typename At, typename Mt, typename Ot >
std::string Triangulation<Vt, At, Mt, Ot>::toString(const Vt& v, At& acc)
{
  return top10::util::toStringD(acc.getX(v))+", "+
      top10::util::toStringD(acc.getY(v))+", "+
      top10::util::toStringD(acc.getZ(v));
}

template< typename Vt, typename At, typename Mt, typename Ot >
Triangulation<Vt, At, Mt, Ot>::UnknownVertex::UnknownVertex(const Point& v):
    top10::util::Error("Triangulation generated a vertex not in the input: "+top10::util::toStringD(v.x())
    +", "+top10::util::toStringD(v.y()))
{
}

template< typename Vt, typename At, typename Mt, typename Ot >
Triangulation<Vt, At, Mt, Ot>::ConflictingConstraint::ConflictingConstraint(const Vt& c1v1, const Vt& c1v2, const Vt& c2v1, const Vt& c2v2):
top10::util::Error("Conflicting constraints"),
c1v1(c1v1), c1v2(c1v2), c2v1(c2v1), c2v2(c2v2)
{
}
