/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/
#include "Outline.hh"
#include "util/HashMap.hh"

namespace
{



  struct IdxEdge
  {
    IdxEdge(unsigned int p1, unsigned int p2): m_p1(p1), m_p2(p2), m_inverted(false)
    {
      if (m_p1 < m_p2)
      {
	std::swap(m_p1, m_p2);
	m_inverted = true;
      }
    }

    unsigned int m_p1;
    unsigned int m_p2;
    bool m_inverted;

    top10::math::Outline::Edge toEdge(const top10::math::Mesh& mesh) const
    {
      unsigned int idx1(m_p1);
      unsigned int idx2(m_p2);
      if (m_inverted)
	std::swap(idx1, idx2);

      const top10::math::Vector p1 = mesh.getVertices()->at(idx1);
      const top10::math::Vector p2 = mesh.getVertices()->at(idx2);

      return top10::math::Outline::Edge(p1, p2);
    }
  };



  struct IdxEdgeHash
  {
    inline size_t operator()(const IdxEdge& e) const
    {
      return 31*e.m_p1 + e.m_p2; 
    }
  };



  struct IdxEdgeCmp
  {
    inline bool operator()(const IdxEdge& e1, const IdxEdge& e2) const
    {
      if (e1.m_p1 < e2.m_p1)
	return true;
      if (e1.m_p1 > e2.m_p1)
	return false;

      return e1.m_p2 < e2.m_p2;
    }
  };


  typedef top10::util::HashMap<IdxEdge, int, IdxEdgeHash, IdxEdgeCmp>::Type IdxEdgeCount;



  void incr(IdxEdgeCount& m, IdxEdge e)
  {
    IdxEdgeCount::iterator f = m.find(e);
    if (f == m.end())
      m[e] = 1;
    else
      ++(f->second);
  }
}

namespace top10 {
namespace math {

const double Outline::EPS = 0.01;

Outline::Outline(const Mesh* m): mesh(m)
{
}


std::list<Outline::Edge> Outline::compute(Vector viewer) const
{
  /* Edges belonging to back faces
    The int counts the number of times an edge appears.
    If it appears only once, it belongs to the outline
  */
  IdxEdgeCount back;
  // Edges in the outline
  std::list<Edge> edges;

  const std::vector<Vector>* vertices = mesh->getVertices();
  const std::vector<Mesh::Face>* faces = mesh->getFaces();
  
  for (std::vector<Mesh::Face>::const_iterator face_it = faces->begin(); face_it != faces->end(); ++face_it)
  {
    // Vector from the viewer to a point of the face
    Vector v = vertices->at(face_it->idxs[0]) - viewer;
    // Compute normal to the face
    Vector n = (vertices->at(face_it->idxs[1]) - vertices->at(face_it->idxs[0])) ^
        (vertices->at(face_it->idxs[2]) - vertices->at(face_it->idxs[1]));
    double vn = v*n;

    if (vn < 0.0)
    {
      incr(back, IdxEdge(face_it->idxs[0], face_it->idxs[1]));
      incr(back, IdxEdge(face_it->idxs[1], face_it->idxs[2]));
      incr(back, IdxEdge(face_it->idxs[2], face_it->idxs[0]));
    }
  }
  
  // Build the list of edges belonging to the outline

  for (IdxEdgeCount::const_iterator it = back.begin(); it != back.end(); ++it)
  {
    if (it->second == 1)
      edges.push_back(it->first.toEdge(*mesh));
  }

  return edges;    
}


std::list<top10::math::Vector>
Outline::sort(const std::list<Edge>& edges0)
{
  std::list<Edge> edges(edges0);
  std::list<top10::math::Vector> ret;
  
  if (edges.empty())
    return ret;

  Edge e = edges.front();
  edges.pop_front();
  ret.push_back(e.p1);
  ret.push_back(e.p2);

  while (!edges.empty())
  {
    std::list<Edge>::iterator closest = edges.begin();
    double d1 = (closest->p1 - ret.back()).size2();
    double d2 = (closest->p2 - ret.back()).size2();
    bool inverted = d2 < d1;
    double min_d = inverted? d2 : d1;
    
    std::list<Edge>::iterator it = closest;
    ++it;
    for (;it != edges.end(); ++it)
    {
      d1 = (it->p1 - ret.back()).size2();
      d2 = (it->p2 - ret.back()).size2();
      bool inv2 = d2 < d1;
      double min_d2 = inv2? d2 : d1;
      if (min_d2 < min_d)
      {
	closest = it;
	min_d = min_d2;
      }
    }
    
    if (min_d != 0.0)
      std::cerr << "Outline::sort: min_d = " << min_d << std::endl;

    if (inverted)
    {
      ret.push_back(closest->p2);
      ret.push_back(closest->p1);
    }
    else
    {
      ret.push_back(closest->p1);
      ret.push_back(closest->p2);
    }
    edges.erase(closest);
  }

  return ret;
}



void Outline::simplify(std::list<top10::math::Vector>* loop, double mindist2)
{
  if (mindist2 <= 0.0) return;

  std::list<top10::math::Vector>::iterator it = loop->begin();
  std::list<top10::math::Vector>::iterator it2;
  std::list<top10::math::Vector>::iterator it3;
  while (it != loop->end())
  {
    it2 = it;
    ++it2;

    if (it2 != loop->end() && (*it - *it2).size2() < mindist2)
    {
      *it2 = 0.5*(*it2 + *it);
      loop->erase(it);
    }

    it = it2;
  }
}

}
}
