/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/
#include <iostream>
#include <sstream>
#include <cmath>

template<int n>
top10::math::Matrix<n>::Matrix()
{
  for (int i=0; i<n; ++i) {
    for (int j=0; j<n; ++j) {
      v[i][j] = 0.0;
    }
  }
}

template<int n>
top10::math::Matrix<n>::Matrix(const double p[n][n])
{
  for (int i=0; i<n; ++i) {
    for (int j=0; j<n; ++j) {
      v[i][j] = p[i][j];
    }
  }
}

template<int n>
top10::math::Matrix<n>::Matrix(const float p[n][n])
{
  for (int i=0; i<n; ++i) {
    for (int j=0; j<n; ++j) {
      v[i][j] = p[i][j];
    }
  }
}

template<int n>
top10::math::Matrix<n>& top10::math::Matrix<n>::operator*=(double x)
{
  for (int i=0; i<n; ++i) {
    for (int j=0; j<n; ++j) {
      v[i][j] *= x;
    }
  }
  return *this;
}

template<int n>
top10::math::Matrix<n>& top10::math::Matrix<n>::operator/=(double x)
{
  for (int i=0; i<n; ++i) {
    for (int j=0; j<n; ++j) {
      v[i][j] /= x;
    }
  }
  return *this;
}

template<int n>
top10::math::Matrix<n>& top10::math::Matrix<n>::operator*=(const top10::math::Matrix<n>& m)
{
  *this = *this * m;
  return *this;
}

template<int n>
top10::math::Matrix<n>& top10::math::Matrix<n>::operator+=(const top10::math::Matrix<n>& m)
{
  for (int i=0; i<n; ++i) {
    for (int j=0; j<n; ++j) {
      v[i][j] += m(i, j);
    }
  }
  return *this;
}

template<int n>
top10::math::Matrix<n>& top10::math::Matrix<n>::operator-=(const top10::math::Matrix<n>& m)
{
  for (int i=0; i<n; ++i) {
    for (int j=0; j<n; ++j) {
      v[i][j] -= m(i, j);
    }
  }
  return *this;
}

template<int n>
top10::math::Matrix<n>& top10::math::Matrix<n>::operator=(const top10::math::Matrix<n>& m)
{
  for (int i=0; i<n; ++i) {
    for (int j=0; j<n; ++j) {
      v[i][j] = m(i, j);
    }
  }
  return *this;
}

template<int n>
bool top10::math::Matrix<n>::operator==(const top10::math::Matrix<n>& m) const
{
  for(int i=0; i<n; ++i) {
    for (int j=0; j<n; ++j) {
      if (v[i][j] != m.v[i][j]) return false;
    }
  }
  return true;    
}

template<int n>
top10::math::Matrix<n> top10::math::operator*(double x, const top10::math::Matrix<n>& m)
{
  top10::math::Matrix<n> N(m);
  return N *= x;
}

template<int n>
top10::math::Matrix<n> top10::math::operator/(const top10::math::Matrix<n>& m, double x)
{
  top10::math::Matrix<n> N(m);
  return N /= x;
}

template<int n>
top10::math::Matrix<n> top10::math::operator+(const top10::math::Matrix<n>& m, const top10::math::Matrix<n>& N)
{
  top10::math::Matrix<n> p(m);
  return p+=N;
}

template<int n>
top10::math::Matrix<n> top10::math::operator-(const top10::math::Matrix<n>& m, const top10::math::Matrix<n>& N)
{
  top10::math::Matrix<n> p(m);
  return p-=N;
}

template<int n>
top10::math::Matrix<n> top10::math::operator*(const top10::math::Matrix<n>& m, const top10::math::Matrix<n>& N)
{
  top10::math::Matrix<n> p;
  for (int i=0; i<n; ++i) {
    for (int j=0; j<n; ++j) {
      for (int k=0; k<n; ++k) {
	p(i, j) += m(i, k)*N(k, j);
      }
    }
  }
  return p;
}

template<int n>
void top10::math::Matrix<n>::dumpTo(std::ostream& str) const
{
  for (int i=0; i<n; ++i) {
    for (int j=0; j<n; ++j) {
      output<double>(str, v[i][j]);
    }
  }
}

template<int n>
void top10::math::Matrix<n>::loadFrom(std::istream& str)
{
  for (int i=0; i<n; ++i) {
    for (int j=0; j<n; ++j) {
      input<double>(str, v[i][j]);
    }
  }
}

template<int n>
top10::math::Matrix<n> top10::math::Matrix<n>::transpose() const
{
  Matrix<n> ret;
  for (int i=0; i<n; ++i) {
    for (int j=0; j<n; ++j) {
      ret(i,j) = operator()(j,i);
    }
  }

  return ret;
}

template<int n>
std::ostream& operator<<(std::ostream& out, const top10::math::Matrix<n>& M)
{
  for (int i=0; i<n; ++i) {
    for (int j=0; j<n; ++j) {
      out<<" "<<M(i,j);
    }
    out<<std::endl;
  }
  return out;
}

template<int n>
std::istream& operator>>(std::istream& in, top10::math::Matrix<n>& M)
{
  for (int i=0; i<n; ++i) {
    for (int j=0; j<n; ++j) {
      in>>M(i,j);
    }
  }
  return in;
}

template<int n>
top10::math::NonInvertible<n>::NonInvertible(const top10::math::Matrix<n>& M)
{
  std::ostringstream buf;
  buf<<M<<" is non invertible";
  std::string::operator=(buf.str());
}

namespace top10 {
namespace math {
    
template<int n>
  int loadXml(const TiXmlElement* el, Matrix<n>* M)
{
  Matrix<n> res;
  for (const TiXmlElement* cell = el->FirstChildElement("cell"); cell; cell = cell->NextSiblingElement("cell"))
  {
    int x=0, y=0;
    int s;
    s = cell->QueryIntAttribute("x", &x);
    if (s<0) return s;
    s = cell->QueryIntAttribute("y", &y);
    if (s<0) return s;
    double val = 0.0;
    s = cell->QueryDoubleAttribute("value", &val);
    if (s<0) return s;
    
    res(x,y) = val;
  }
  assert(M);
  *M = res;
  return 0;
}
    
template<int n>
  int saveXml(TiXmlElement* el, const Matrix<n>& M)
{
  for (int i=0; i<n; ++i) {
    for (int j=0; j<n; ++j) {
      if (M(i,j) != 0.0) {
        TiXmlElement cell("cell");
        cell.SetAttribute("x", i);
        cell.SetAttribute("y", j);
        cell.SetDoubleAttribute("value", M(i,j));
        el->InsertEndChild(cell);
      }
    }
  }
  return 0;
}

}
}
