/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_MATHHERMITE_HH
#define TOP10_MATHHERMITE_HH

#include "Vertex.hh"
#include <vector>
#include <map>

namespace top10 {
namespace math {

//! A Hermite curve in 3d space.
/*! A Hermite curve is parameterized by its endpoints and the tangents at the ends. */
class Hermite {
public:
  Hermite();

  inline void setEnd1(Vector P)      { P1 = P; }
  inline void setEnd2(Vector P)      { P4 = P; }
  inline void setTangent1(Vector R)  { R1 = R; }
  inline void setTangent2(Vector R)  { R4 = R; }
  
  inline Vector getEnd1() const     { return P1; }
  inline Vector getEnd2() const     { return P4; }
  inline Vector getTangent1() const { return R1; }
  inline Vector getTangent2() const { return R4; }
  
  //! Evaluate the curve at some point t, 0<=t<=1
  Vector eval(float t) const;
  
  //! Evaluate the tangent at some point t, 0<=t<=1
  /*! \return the non-normalized tangent */
  Vector eval_tg(float t) const;
  
  //! Try to find the closest point to p on the curve.
  /*! \return true if succedded. In that case, t_out is set with the result. */
  bool findClosest(Vector p, float& t_out) const;
  
  //! Return true if there exist t1 and t2, 0 <= t1 < t2 <= 1 such that H(t1) = H(t2)
  bool hasLoop() const;
  //! 2D version of hasLoop: H(t1) = H(t2) is relaxed to H(t1).y = H(t2).y and H(t1).z = H(t2).z
  bool hasLoopX() const;
  bool hasLoopY() const;
  bool hasLoopZ() const;

  //! Compute a vector of vertices on the curve
  /*! The vertices are computed so that the maximum angle between two consecutive edges is bounded by max_angle
    \param max_angle The maximum angle between two edges (in rads)
    \param min_dist Square value of the minimum size of an edge. Prevails over max_angle.
    \param out Output vector where the vertices are inserted (at the end)
    \param t_out optional pointer to a vector that will hold the values of t.
    \return the number of vertices computed
  */
  int vectorize(double max_angle, double min_dist, std::vector<Vector>& out, std::vector<float>* t_out = 0) const;
  
  //! Vectorize for t: t_start <= t <= t_end
  int vectorize(double max_angle, double min_dist, std::vector<Vector>& out, double t_start, double t_end, std::vector<float>* t_out = 0) const;

  //! Return the length of the curve between two given points
  double getLength(double t_start, double t_end, double t_step = 0.1) const;

private:
  //! Endpoints
  Vector P1, P4;
  //! Tangents
  Vector R1, R4;
  
  //! Refine the vectorization (called by vectorize)
  /*! Requires that t_to_pt contains an entry for t, 0 and 1, and that 0<t<1 */
  void refine(double max_angle, double min_dist, std::map<float, Vector>& t_to_pt, float t) const;
};

}
}

#endif
