/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/
#include "Hermite.hh"
#include "util/Log.hh"

namespace top10 {
namespace math {

Hermite::Hermite():
  P1(0.0,0.0,0.0), P4(1.0,0.0,0.0), R1(1.0,0.0,0.0), R4(-1.0,0.0,0.0)
{
}

Vector Hermite::eval(float t) const
{
  Vector X = (2*P1 - 2*P4 + R1 + R4)*t;
  X = (X - 3*P1 + 3*P4 - 2*R1 - R4)*t;
  X = (X + R1)*t + P1;
  return X;
}

Vector Hermite::eval_tg(float t) const
{
  Vector X = 3*(2*P1 - 2*P4 + R1 + R4)*t;
  X = (X + 2*(-3*P1 + 3*P4 - 2*R1 - R4))*t;
  X = X + R1;
  return X;  
}

int Hermite::vectorize(double max_angle, double min_dist, std::vector<Vector>& out, std::vector<float>* t_out) const
{
  return vectorize(max_angle, min_dist, out, 0.0, 1.0, t_out);
}

int Hermite::vectorize(double max_angle, double min_dist, std::vector<Vector>& out, double t_start, double t_end, std::vector<float>* t_out) const
{
  double t1 = (3*t_start+t_end)/4;
  double t2 = (t_start + t_end)/2;
  double t3 = (t_start+3*t_end)/4;
  
  std::map<float, Vector> t_to_pt;
  t_to_pt[t_start] = eval(t_start);
  t_to_pt[t1] = eval(t1);
  t_to_pt[t2] = eval(t2);
  t_to_pt[t3] = eval(t3);
  t_to_pt[t_end] = eval(t_end);
  
  refine(max_angle, min_dist, t_to_pt, t1);
  refine(max_angle, min_dist, t_to_pt, t2);
  refine(max_angle, min_dist, t_to_pt, t3);
  
  for (std::map<float, Vector>::const_iterator it = t_to_pt.begin(); it != t_to_pt.end(); ++it) {
    if (t_out) t_out->push_back(it->first);
    out.push_back(it->second);
  }
  
  return t_to_pt.size();
}

void Hermite::refine(double max_angle, double min_dist, std::map<float, Vector>& t_to_pt, float t) const
{
  // Look for the points just before and just after t
  std::map<float, Vector>::const_iterator it0, it1, it;
  it = t_to_pt.find(t);
  assert(it != t_to_pt.begin());
  it0 = it; --it0;
  assert(it != t_to_pt.end());
  it1 = it; ++it1;
  assert(it1 != t_to_pt.end());
  
  // Compute the directions
  Vector dir0 = it->second - it0->second;
  Vector dir1 = it1->second - it->second;
  double s_dir = dir0.size();
  if (s_dir < SMALL_VECTOR) return;
  dir0 /= s_dir;
  s_dir = dir1.size();
  if (s_dir < SMALL_VECTOR) return;
  dir1 /= s_dir;
  
  // Compute the angle between dir0 and dir1
  double cosval = (dir0*dir1);
  double angle = acos(cosval);
  if (max_angle < angle) {
    // Compute the distance^2
    Vector dist = it->second - it0->second;
    if (dist.size2() > min_dist) {
      double t2 = 0.5*(it0->first+t);
      Vector v = eval(t2);
      t_to_pt[t2] = v;
      refine(max_angle, min_dist, t_to_pt, t2);
    }
    
    dist = it1->second - it->second;
    if (dist.size2() > min_dist) {
      double t2 = 0.5*(t+it1->first);
      Vector v = eval(t2);
      t_to_pt[t2] = v;
      refine(max_angle, min_dist, t_to_pt, t2);
    }
  }
}

double Hermite::getLength(double t_start, double t_end, double t_step) const
{
  double l = 0;
  Vector prev = eval(t_start);
  for (double t = t_start; t <= t_end; t += t_step)
  {
    Vector curr = eval(t);
    l += (curr - prev).size();
    prev = curr;
  }
  return l;
}

}
}
