/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "Frustum.hh"
#include "Mesh.hh"
#include "intersections.hh"
#include <cassert>

using namespace top10::math;

Frustum::Frustum(): center(0,0,0), e0(1,0,0), e1(0,1,0), e2(0,0,1), ratio(1.0), fov(M_PI/3.0), dist_near(1.0), dist_far(1e3)
{
}

void Frustum::setDirection(Vector dir, Vector up)
{
  if (dir.size() < SMALL_VECTOR || (dir^up).size() < SMALL_VECTOR) return;

  e2 = -1.0*dir;
  double s=e2.size();
  e2/=s;

  e0 = up^e2;
  s = e0.size();
  e0/=s;

  e1 = e2^e0;
}

void Frustum::getGLMatrix(double out[4][4]) const
{
  Vector e[3] = {e0, e1, e2};
  top10::math::Matrix3 M3(e);
  top10::math::Matrix4 M4(M3);
  M4 = M4.transpose();
  top10::math::Translation4 T4(-center);
  M4 = M4*T4;
  M4.toGL(out);
}

void Frustum::getOrientGLMatrix(double out[4][4]) const
{
  Vector e[3] = {e0, e1, e2};
  top10::math::Matrix3 M3(e);
  top10::math::Matrix4 M4(M3);
  M4 = M4.transpose();
  M4.toGL(out);
}

void Frustum::update()
{
  // Compute the vertices of the view volume
  double h = fabs(tan(fov)/2.0);
  double w = h*ratio;
  vertices[nul] = center+dist_near*(-e0*w+e1*h-e2);
  vertices[nur] = center+dist_near*(e0*w+e1*h-e2);
  vertices[nll] = center+dist_near*(-e0*w-e1*h-e2);
  vertices[nlr] = center+dist_near*(e0*w-e1*h-e2);
  vertices[ful] = center+dist_far*(-e0*w+e1*h-e2);
  vertices[fur] = center+dist_far*(e0*w+e1*h-e2);
  vertices[fll] = center+dist_far*(-e0*w-e1*h-e2);
  vertices[flr] = center+dist_far*(e0*w-e1*h-e2);

  planes[left] = Plane(vertices[fll], vertices[ful], center);
  planes[right] = Plane(vertices[fur], vertices[flr], center);
  planes[upper] = Plane(vertices[ful], vertices[fur], center);
  planes[lower] = Plane(vertices[flr], vertices[fll], center);
  planes[far_plane] = Plane(center-e2*dist_far, e2);
  planes[near_plane] = Plane(center-e2*dist_near, -e2);

  // transform this frustum to polygons
  Mesh new_polygons;
  new_polygons.setVertices(8, vertices);
  
  // Left
  new_polygons.addFace(nul, nll, fll, ful);

  // Right
  new_polygons.addFace(nur, nlr, flr, fur);

  // Upper
  new_polygons.addFace(nur, nul, ful, fur);

  // Lower
  new_polygons.addFace(nlr, nll, fll, flr);

  // Far
  new_polygons.addFace(flr, fur, ful, fll);

  // Near
  new_polygons.addFace(nlr, nur, nul, nll);

  polygons = new_polygons;    
}

top10::helpers::OverlapType Frustum::overlap(const AxisAlignedBox& box) const
{
  if (intersect(polygons, box))
    return top10::helpers::Intersect;

  // No intersection with the polygons: check if the box is inside the frustum
  Vector vertices[8];
  box.getVertices(vertices);
  for (int i=0; i<8; ++i) {
     if (planes[left].isBelow(vertices[i]) && planes[right].isBelow(vertices[i]) &&
	 planes[upper].isBelow(vertices[i]) && planes[lower].isBelow(vertices[i]) &&
	 planes[far_plane].isBelow(vertices[i]) && planes[near_plane].isBelow(vertices[i]))
	 return top10::helpers::Contains; 
  }
  
  // Check if the frustum is inside the box
  for (int i=0; i<8; ++i)
    if (box.isInside(this->vertices[i]))
      return top10::helpers::Contained;

  return top10::helpers::None;
}

top10::helpers::OverlapType Frustum::overlap(const Box& box) const
{
  if (intersect(polygons, box))
    return top10::helpers::Intersect;

  // No intersection with the polygons: check if the box is inside the frustum
  Vector vertices[8];
  box.getVertices(vertices);
  for (int i=0; i<8; ++i) {
     if (planes[left].isBelow(vertices[i]) && planes[right].isBelow(vertices[i]) &&
	 planes[upper].isBelow(vertices[i]) && planes[lower].isBelow(vertices[i]) &&
	 planes[far_plane].isBelow(vertices[i]) && planes[near_plane].isBelow(vertices[i]))
	 return top10::helpers::Contains; 
  }
  
  // Check if the frustum is inside the box
  for (int i=0; i<8; ++i)
    if (box.isInside(this->vertices[i]))
      return top10::helpers::Contained;

  return top10::helpers::None;
}

top10::math::Matrix3 Frustum::getOrientInv() const
{
  Vector e[3] = {e0, e1, e2};
  top10::math::Matrix3 M3(e);
  return M3.transpose();
}

top10::math::Matrix3 Frustum::getOrient() const
{
  Vector e[3] = {e0, e1, e2};
  top10::math::Matrix3 M3(e);
  return M3;
}
