/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "Curve.hh"

using namespace top10::math;

double Curve2D::getY(double x) const
{
  std::map<double, double>::const_iterator b = ctrl_points.upper_bound(x);
  if (b == ctrl_points.end()) {
    if (!ctrl_points.empty())
    {
      b--;
      return b->second;
    }
    else
      return 0.0; // Empty curve is the flat curve y=0
  }
  else if (b == ctrl_points.begin()) {
    return b->second;
  }
  else {
    std::map<double, double>::const_iterator a = b; a--;
    return (x - a->first)*(b->second - a->second)/(b->first - a->first) + a->second;
  }
}

bool Curve2D::addXY(double x, double y)
{
  return ctrl_points.insert(std::pair<double, double>(x,y)).second;
}

bool Curve2D::removeXY(double x)
{
  return ctrl_points.erase(x) == 1;
}

void Curve2D::dumpTo(std::ostream& str) const
{
  for (std::map<double, double>::const_iterator p = ctrl_points.begin();
       p != ctrl_points.end();
       ++p) {
    output<double>(str, p->first);
    output<double>(str, p->second);
  }
}

void Curve2D::loadFrom(std::istream& str)
{
  for (std::map<double, double>::iterator p = ctrl_points.begin();
       p != ctrl_points.end();
       ++p) {
    double first;
    input<double>(str, first);
    if (first != p->first) throw ParseError("Can not change the control points of a Curve2D in loadFrom()");
    input<double>(str, p->second);
  }
}

void Curve2D::getPoints(std::list<std::pair<double,double> > *out_list) const
{
  for (std::map<double, double>::const_iterator it = ctrl_points.begin();
    it != ctrl_points.end();
    ++it)
  {
    out_list->push_back(*it);
  }
}

int Curve2D::loadXml(const TiXmlElement* node)
{
  int status = 0;

  Curve2D new_one;

  const TiXmlElement* child = node->FirstChildElement("point");
  while (child)
  {
    float x, y;
    int status2 = child->QueryFloatAttribute("x", &x);
    if (status2 != 0)
    {
      status = status2;
      logXmlNodeError("Missing or wrong attribute 'x'", child, top10::util::Log::Error);
    }

    status2 = child->QueryFloatAttribute("y", &y);
    if (status2 != 0)
    {
      status = status2;
      logXmlNodeError("Missing or wrong attribute 'y'", child, top10::util::Log::Error);
    }

    new_one.addXY(x,y);
    child = child->NextSiblingElement("point");
  }

  ctrl_points.swap(new_one.ctrl_points);

  return status;
}

int Curve2D::saveXml(TiXmlElement* node) const
{
  for (std::map< double, double >::const_iterator it = ctrl_points.begin();
    it != ctrl_points.end();
    ++it)
  {
    TiXmlElement child("point");
    child.SetDoubleAttribute("x", it->first);
    child.SetDoubleAttribute("y", it->second);
    node->InsertEndChild(child);
  }

  return 0;
}

// ****************************************************************************

PointND CurveND::get(double t) const
{
  const_iterator b = upper_bound(t);
  if (b == end()) {
    b--;
    return b->second;
  }
  else if (b == begin()) {
    return b->second;
  }
  else {
    const_iterator a = b; a--;
    PointND ret;

    ret = (t - a->first)*(b->second - a->second)/(b->first - a->first) + a->second;

    return ret;
  }
}

bool CurveND::add(double t, const PointND& v)
{
  return insert(std::pair<double, PointND>(t,v)).second;
}
