/*
  Top10, a racing simulator
  Copyright (C) 2006  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "initGL.hh"
#include "util/GlobalOptions.hh"
#include "util/error.hh"

#include <vector>
#include <string>

namespace top10 {
namespace helpers {

  struct BitSizes {
    BitSizes(int s, int r, int g, int b, int d): stencil(s), red(r), green(g), blue(b), depth(d) {}

    int stencil;
    int red;
    int green;
    int blue;
    int depth;

    void dump() const
      {
	std::cout<<"S:" << stencil<< " R:" << red <<" G:" << green << " B:" << blue << " D:" << depth;
      }
  };

SDL_Surface* initGL()
{
  int w = getOptI("Render.Window.Width");
  int h = getOptI("Render.Window.Height");
  bool full_screen = getOptI("Render.Window.Fullscreen") != 0;
  return initGL(w, h, full_screen);
}

SDL_Surface* initGL(int w, int h, bool full_screen)
{  
  // Try the following combinations in the order specified
  std::vector<BitSizes> bits;
  bits.push_back( BitSizes(8, 8, 8, 8, 32) );
  bits.push_back( BitSizes(8, 8, 8, 8, 16) );
  bits.push_back( BitSizes(8, 5, 6, 5, 16) );

  using top10::util::Error;

  SDL_Surface* screen;

  for (std::vector<BitSizes>::const_iterator it = bits.begin(); it != bits.end(); ++it)
  {
    SDL_GL_SetAttribute( SDL_GL_RED_SIZE, it->red);
    SDL_GL_SetAttribute( SDL_GL_GREEN_SIZE, it->green );
    SDL_GL_SetAttribute( SDL_GL_BLUE_SIZE, it->blue );
    SDL_GL_SetAttribute( SDL_GL_DEPTH_SIZE, it->depth);
    SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );
    SDL_GL_SetAttribute( SDL_GL_STENCIL_SIZE, it->stencil );
    std::cout << "Trying ";
    it->dump();
    screen = SDL_SetVideoMode( w, h, 0, SDL_OPENGLBLIT | (full_screen?SDL_FULLSCREEN:0));
    std::cout << std::endl;
    if (screen) return screen;
  }

  if (screen == NULL) {
    throw Error(std::string("Could not set GL mode: ") + std::string(SDL_GetError()));
  }

  return NULL;
}

void toggleGrabInput()
{
  SDL_GrabMode mode = SDL_WM_GrabInput(SDL_GRAB_QUERY);
  setGrabInput(mode == SDL_GRAB_OFF);
}

void setGrabInput(bool b)
{
  if (!b)
  {
    SDL_WM_GrabInput(SDL_GRAB_OFF);
    SDL_ShowCursor(SDL_ENABLE);
  }
  else
  {
    SDL_WM_GrabInput(SDL_GRAB_ON);
    SDL_ShowCursor(SDL_DISABLE);
  }
}


SDL_Surface* toggleFullscreen(SDL_Surface* surface)
{
  SDL_Surface* new_surface =0;
  int w = surface->w;
  int h = surface->h;
  Uint8 bpp = surface->format->BitsPerPixel;

  Uint32 flags = surface->flags;
  flags = flags ^ SDL_FULLSCREEN;

  if (SDL_VideoModeOK(w, h, bpp, flags))
    new_surface = SDL_SetVideoMode(w, h, bpp, flags);

  if (new_surface)
  {
    SDL_FreeSurface(surface);
    return new_surface;
  }
  else
    return surface;
}

};
};
