/*
  Top10, a racing simulator
  Copyright (C) 2006  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "initAL.hh"
#include "util/Log.hh"

using top10::util::Log;

namespace top10
{
namespace helpers
{
  AudioContext* AudioContext::single = 0;
  AudioContext* AudioContext::getSingle()
  {
    return getSingle(0);
  }

  AudioContext* AudioContext::getSingle(const char* device_name)
  {
    if (single)
      return single;

    ALenum al_err;
    al_err = alGetError();
    if (al_err != AL_NO_ERROR)
      Log::getSingle()->send(Log::Warning, "AudioContext/getSingle",
      "AL error " + makeErrorStr(al_err) + " before alcOpenDevice.");

    ALCdevice* dev = alcOpenDevice(device_name);

    al_err = alGetError();
    if (al_err != AL_NO_ERROR)
      Log::getSingle()->send(Log::Warning, "AudioContext/getSingle",
      "AL error " + makeErrorStr(al_err) + " after alcOpenDevice.");

    if (dev)
    {
      single = new AudioContext;
      single->dev = dev;
      
      single->context = alcCreateContext(dev, 0);

      al_err = alGetError();
      if (al_err != AL_NO_ERROR)
	Log::getSingle()->send(Log::Warning, "AudioContext/getSingle",
	"AL error " + makeErrorStr(al_err) + " after alcCreateContext.");

      if (single->context)
	alcMakeContextCurrent(single->context);

      al_err = alGetError();
      if (al_err != AL_NO_ERROR)
	Log::getSingle()->send(Log::Warning, "AudioContext/getSingle",
	"AL error " + makeErrorStr(al_err) + " after alcMakeContextCurrent.");

    }
    return single;
  }

  AudioContext::AudioContext()
    : dev(0),
    context(0)
  {
  }

  AudioContext::~AudioContext()
  {
    alcDestroyContext(context);
    alcCloseDevice(dev);
  }

  AudioContext* initAL(const char* device_name)
  {
    return AudioContext::getSingle(device_name);
  }

  AudioContext* initAL()
  {
    return initAL(0);
  }

  void closeAL(AudioContext* context)
  {
    if (context)
      delete context;
  }

  std::string makeErrorStr(ALenum err)
  {
    switch(err)
    {
    case AL_NO_ERROR:          return std::string("AL_NO_ERROR");          break;
    case AL_INVALID_NAME:      return std::string("AL_INVALID_NAME");      break;
    case AL_INVALID_ENUM:      return std::string("AL_INVALID_ENUM");      break;
    case AL_INVALID_VALUE:     return std::string("AL_INVALID_VALUE");     break;
    case AL_INVALID_OPERATION: return std::string("AL_INVALID_OPERATION"); break;
    case AL_OUT_OF_MEMORY:     return std::string("AL_OUT_OF_MEMORY");     break;
    default:                   return std::string("AL_???");               break;
    }
    return std::string("<UNREACHABLE>");
  }

};
};