/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/
#include "ReadCal3d.hh"
#include "graphX/MaterialNode.hh"
#include "graphX/MeshNode.hh"
#include "math/Mesh.hh"

namespace top10 {

namespace helpers {

ReadCal3d::ReadCal3d(std::string filename):
  m_calCoreModel(0),
  m_scale(1.0)
{
  finder = top10::util::PathFinder::defaultPathFinder();
  std::string path = finder.find(filename);
  if (path.empty()) throw std::string("Could not find ")+filename;
  std::string dir = top10::util::PathFinder::getPath(path);
  finder.addPath(dir);
  
  m_calCoreModel = new CalCoreModel("Cal3d imported model");
  
  std::ifstream in(path.c_str());
  if (!in) throw std::string("Failed to open "+path);
  
  parseCfg(in);
  
  import();
}

ReadCal3d::~ReadCal3d()
{
  delete m_calCoreModel;
}

void ReadCal3d::parseCfg(std::ifstream& file)
{
    // parse all lines from the model configuration file
  int line;
  for(line = 1; ; line++)
  {
    // read the next model configuration line
    std::string strBuffer;
    std::getline(file, strBuffer);

    // stop if we reached the end of file
    if(file.eof()) break;

    // check if an error happend while reading from the file
    if(!file)
      throw std::string("Error while reading from the model configuration file");

    // Used to store positions returned by std::string::find and friends.
    std::string::size_type find_pos;

    // find the first non-whitespace character
    std::string::size_type pos;
    pos = strBuffer.find_first_not_of(" \t");

    // check for empty lines
    if((pos == std::string::npos) || (strBuffer[pos] == '\n') || (strBuffer[pos] == '\r') || (strBuffer[pos] == 0)) continue;

    // check for comment lines
    if(strBuffer[pos] == '#') continue;

    // get the key
    std::string strKey;
    find_pos = strBuffer.find_first_of(" =\t\n\r", pos);
    if (find_pos == std::string::npos)
    {
      std::cerr << "(" << line << "): Invalid syntax. Ignoring line." << std::endl;
      continue;
    }

    strKey = strBuffer.substr(pos, find_pos - pos);
    pos += strKey.size();

    // get the '=' character
    find_pos = strBuffer.find_first_not_of(" \t", pos);
    if((pos == std::string::npos) || (strBuffer[pos] != '='))
    {
      std::cerr << "(" << line << "): Invalid syntax. Ignorind line." << std::endl;
      continue;
    }
    pos = find_pos;

    // find the first non-whitespace character after the '=' character
    ++pos;
    if (pos >= strBuffer.size())
    {
      std::cerr << "(" << line << "): Invalid syntax. Ignoring line." << std::endl;
      continue;
    }

    find_pos = strBuffer.find_first_not_of(" \t", pos);
    if (find_pos == std::string::npos)
    {
      std::cerr << "(" << line << "): Invalid syntax. Ignoring line." << std::endl;
      continue;
    }
    pos = find_pos;

    // get the data
    std::string strData;
    find_pos = strBuffer.find_first_of("\r\n");
    if (find_pos == std::string::npos)
      find_pos = strBuffer.size();
    strData = strBuffer.substr(pos, find_pos - pos);

    // handle the model creation
    if(strKey == "scale")
    {
      // set rendering scale factor
      m_scale = atof(strData.c_str());
    }
    else if(strKey == "skeleton")
    {
      // load core skeleton
      std::cout << "Loading skeleton '" << strData << "'..." << std::endl;
      std::string path = finder.find(strData);
      if (path.empty()) std::cerr<<"Could not find skeleton "<<strData<<std::endl;
      else if(!m_calCoreModel->loadCoreSkeleton(path))
      {
        CalError::printLastError();
      }
    }
    else if(strKey == "animation")
    {
      // load core animation
      std::cout << "Loading animation '" << strData << "'..." << std::endl;
      std::string path = finder.find(strData);
      if (path.empty()) std::cerr<<"Could not find animation "<<strData<<std::endl;
      else if(m_calCoreModel->loadCoreAnimation(path) == -1)
      {
        CalError::printLastError();
      }
    }
    else if(strKey == "mesh")
    {
      // load core mesh
      std::cout << "Loading mesh '" << strData << "'..." << std::endl;
      std::string path = finder.find(strData);
      if (path.empty()) std::cerr<<"Could not find mesh "<<strData<<std::endl;
      else if(m_calCoreModel->loadCoreMesh(path) == -1)
      {
        CalError::printLastError();
      }
    }
    else if(strKey == "material")
    {
      // load core material
      std::cout << "Loading material '" << strData << "'..." << std::endl;
      std::string path = finder.find(strData);
      if (path.empty()) std::cerr<<"Could not find material "<<strData<<std::endl;
      else if(m_calCoreModel->loadCoreMaterial(path) == -1)
      {
        CalError::printLastError();
      }
    }
    else
    {
      // everything else triggers an error message, but is ignored
      std::cerr << "(" << line << "): Invalid syntax." << std::endl;
    }
  }

  // explicitely close the file
  file.close();
}

void ReadCal3d::import()
{
  // Taken from Cal3d miniviewer's code:
  // 
  // make one material thread for each material
  // NOTE: this is not the right way to do it, but this viewer can't do the right
  // mapping without further information on the model etc., so this is the only
  // thing we can do here.
  //
  int materialId;
  for(materialId = 0; materialId < m_calCoreModel->getCoreMaterialCount(); materialId++)
  {
    // create the a material thread
    m_calCoreModel->createCoreMaterialThread(materialId);

    // initialize the material thread
    m_calCoreModel->setCoreMaterialId(materialId, 0, materialId);
  }
  
  // Retrieve the geometry by "rendering" the cal3d model
  CalModel* cal_model = new CalModel(m_calCoreModel);
  
  // attach all meshes to the model
  for(int meshId = 0; meshId < m_calCoreModel->getCoreMeshCount(); meshId++)
  {
    cal_model->attachMesh(meshId);
  }
  
  // set the material set of the whole model
  cal_model->setMaterialSet(0);
  
  // begin the rendering loop
  CalRenderer *pCalRenderer = cal_model->getRenderer();
  if (pCalRenderer->beginRendering()) {
    // get the number of meshes
    int meshCount = pCalRenderer->getMeshCount();

    top_node = new top10::graphX::GroupNode;
    
    // render all meshes of the model
    for(int meshId = 0; meshId < meshCount; meshId++)
    {
      // get the number of submeshes
      int submeshCount = pCalRenderer->getSubmeshCount(meshId);

      // render all submeshes of the mesh
      for(int submeshId = 0; submeshId < submeshCount; submeshId++)
      {
        // select mesh and submesh for further data access
        if(pCalRenderer->selectMeshSubmesh(meshId, submeshId))
        {
          unsigned char meshColor[4];
          
          // set the material diffuse color
          pCalRenderer->getDiffuseColor(&meshColor[0]);
          top10::graphX::MaterialNode* mat_node = new top10::graphX::MaterialNode;
          mat_node->r = meshColor[0];
          mat_node->g = meshColor[1];
          mat_node->b = meshColor[2];
          
          // get the transformed vertices of the submesh
          int vertexCount = pCalRenderer->getVertexCount();
          float* vertex_buff = new float[(1+vertexCount)*3];   // "1+" so that we don't try to allocate 0 bytes
          int vc2 = pCalRenderer->getVertices(vertex_buff);
          assert(vc2 == vertexCount);
          
          // get the normals
          int normalsCount = vertexCount;
          float* normals_buff = new float[(1+normalsCount)*3];
          vc2 = pCalRenderer->getNormals(normals_buff);
          assert(vc2 == normalsCount);
          
          // Get the texture coordinates
          int texelsCount = vertexCount;
          float* tex_buff = new float[(1+texelsCount)*2];
          vc2 = pCalRenderer->getTextureCoordinates(0, tex_buff);
          if (vc2 <= 0) { // Not using texture mapping?
            for (int i=0; i<normalsCount; ++i) tex_buff[2*i] = tex_buff[2*i+1] = 0.0;
          }
          assert(vc2 == 0 || vc2 == texelsCount || (vc2 == -1 && pCalRenderer->getMapCount() == 0));
          
          // Get the faces
          int faceCount = pCalRenderer->getFaceCount();
          CalIndex* face_buff = new CalIndex[(1+faceCount)*3];
          vc2 = pCalRenderer->getFaces(face_buff);
          assert(vc2 == faceCount);
          
          // Create a mesh
          top10::math::Mesh* mesh = new top10::math::Mesh;
          mesh->setVertices(vertexCount, vertex_buff);
          for (int i=0; i<faceCount; ++i) {
            mesh->addFace(face_buff[i*3], face_buff[i*3 +1], face_buff[i*3 +2]);
          }
          mesh->canonizeFaces();

          // Create the MeshNode
          top10::graphX::MeshNode* mesh_node = 
            new top10::graphX::MeshNode(mesh, normalsCount*3, normals_buff, texelsCount*2, tex_buff, m_calCoreModel->getCoreMesh(meshId)->getName());
          
          mat_node->addChild(mesh_node);
          top_node->addChild(mat_node);
          
          delete[] vertex_buff;
          delete[] normals_buff;
          delete[] tex_buff;
          delete[] face_buff;
          
        }
        
      }
    }
    
    // end the rendering
    pCalRenderer->endRendering();
  }
  else {
    CalError::printLastError();
    throw std::string("Succeeded to load the model, but failed to import it");
  }
  
  delete cal_model;
}

}
}
