/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005,2006  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/
#ifndef TOP10_HELPERSMESHEDITOR_HH
#define TOP10_HELPERSMESHEDITOR_HH

#include "math/Matrix.hh"
#include "math/AxisEnum.hh"
#include "util/XmlDumpable.hh"
#include "graphX/TransformNode.hh"

#include <list>
#include <string>

namespace top10 {
namespace helpers {

class MeshEditor: public top10::util::XmlDumpable
{
public:
  //! Flags associated to a mesh.
  enum MeshFlags
  {
    NO_FLAGS = 0,
    FACE_SHOW_FRONT = 1, //!< Show front faces
    FACE_SHOW_BACK = 2,  //!< Show back faces
    SHADOW_VOLUME = 4,   //!< Project a shadow volume
    NO_COLLISION = 8     //!< No collision: immaterial object
  };

public:
  MeshEditor();
  
  /*
   * Editor interface
   */
  //! Get the filenames of all meshes
  std::list<std::string> getMeshNames() const;
  
  //! Create a new entry for a mesh and point on it
  void addMesh();
  //! Delete the current entry
  void delMesh();
  //! Point on a mesh given its name
  void gotoMesh(std::string name);
    
  //! Set the filename of the current mesh and try to load it
  /*! The filename is changed even if the mesh could not be loaded. */
  void loadMesh(std::string filename);
  //! Set the name of the current mesh
  void setName(std::string name);
  //! Set the transformation matrix of the mesh
  void setTransform(const top10::math::Matrix4&);
  //! Translate the current mesh
  void translate(top10::math::Vector);
  //! Rotate the current mesh around the X, Y or Z axis
  void rotate(top10::math::AxisEnum, double angle_degrees);
  //! Set a flag in the current mesh
  void setFlag(MeshFlags);
  //! Clear a flag in the current mesh
  void clearFlag(MeshFlags);

  //! Get the name of the file of the current mesh
  std::string getMeshFilename() const;
  //! Get the name of the current mesh
  std::string getMeshName() const;
  //! Get the transform matrix of the current mesh
  top10::math::Matrix4 getTransform() const;
  //! Get the unique id of the current mesh
  int getId() const;
  //! Get the node of the current mesh
  top10::graphX::Node* getCurrentNode();
  //! Get the flags of the current mesh
  unsigned int getFlags() const;

  //! Get the name of the file of a mesh
  std::string getMeshFilename(int id) const;
  //! Get the name of a mesh
  std::string getMeshName(int id) const;
  //! Get the transform matrix of a mesh
  top10::math::Matrix4 getTransform(int id) const;
  //! Find the id of a mesh
  int findId(std::string name) const;
  //! Get the node of a mesh
  top10::graphX::Node* getNode(int id);
  //! Get the flags of a mesh
  unsigned int getFlags(int id) const;

public:
  //! Base class for all classes which wish to be notified when something changes
  class Listener: public top10::util::RefCount {
  public:
    virtual void notifyTransform(const top10::math::Matrix4&);
    
    virtual ~Listener() {}
  };
  typedef top10::util::Ref< Listener > ListenerRef;
  typedef std::vector< ListenerRef > ListenerRefs;

  void addListener(Listener*);
  void removeListener(Listener*);
  void clearState();

protected:
  /*
   * XmlDumpable implementation
   */
  
  int loadXml(const TiXmlElement*);
  int saveXml(TiXmlElement*) const;
  std::string getOrigin() const;
  
private:
  struct MeshDesc {
    std::string name;
    std::string filename;
    int id;
    top10::util::Ref<top10::graphX::TransformNode> transform_node;
    top10::util::Ref<top10::graphX::Node> mesh_node;
    unsigned int flags;

    void createNode();
  };
  typedef std::list< MeshDesc > MeshList;
  
  MeshList meshes;
  MeshList::iterator current_mesh;
  int next_id;
  ListenerRefs listeners;

  int loadItem(const TiXmlElement*, MeshList*, int*);
  int saveItem(TiXmlElement*, MeshList::const_iterator) const;
  MeshList::iterator find(std::string);
  MeshList::const_iterator find(std::string) const;
  MeshList::iterator find(int);
  MeshList::const_iterator find(int) const;
  void notifyTransform() const;
};

}
}

#endif
