/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005,2006  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/
#include "MeshEditor.hh"
#include "MeshFactory.hh"
#include "util/strconv.hh"
#include "util/findpred.hh"
#include "util/NameFinder.hh"
#include "util/Log.hh"
#include "graphX/GroupNode.hh"
#include "graphX/CullNode.hh"
#include "graphX/NodeIterator.hh"
#include "graphX/MeshNode.hh"

using top10::util::Log;

top10::helpers::MeshEditor::MeshEditor( ):
  top10::util::XmlDumpable("mesh_repository")
{
  clearState();
}

std::list< std::string > top10::helpers::MeshEditor::getMeshNames( ) const
{
  std::list< std::string > ret;
  for (MeshList::const_iterator it = meshes.begin(); it != meshes.end(); ++it)
  {
    ret.push_back(it->name);
  }
  
  return ret;
}

void top10::helpers::MeshEditor::addMesh( )
{
  MeshDesc new_one;
  new_one.id = next_id;
  new_one.transform_node = new top10::graphX::TransformNode;
  new_one.flags = NO_FLAGS;

  // Find an unused name
  new_one.name = 
    top10::util::NameFinder< top10::util::FindNamed >::getNewName(meshes.begin(), meshes.end());
  
  // Add the new entry
  meshes.push_back(new_one);
  
  // Move the pointer to it
  current_mesh = meshes.end(); --current_mesh;
  
  // Increase the "next unique id"
  ++next_id;
}

void top10::helpers::MeshEditor::delMesh( )
{
  if (current_mesh == meshes.end()) return;
  meshes.erase(current_mesh);
  current_mesh = meshes.begin();
}

void top10::helpers::MeshEditor::gotoMesh( std::string name )
{
  MeshList::iterator it = find(name);
  if (it != meshes.end()) current_mesh = it;
}

void top10::helpers::MeshEditor::setName( std::string name )
{
  if (current_mesh != meshes.end()) {
    if (find(name) != meshes.end()) {
      Log::getSingle()->send(Log::Error, getOrigin(), "There is already another mesh by that name");
      return;
    }
    current_mesh->name = name;
  }
}

void top10::helpers::MeshEditor::setTransform( const top10::math::Matrix4& M)
{
  if (current_mesh == meshes.end()) return;
  current_mesh->transform_node->setToWorld(M);
  notifyTransform();
}

void top10::helpers::MeshEditor::translate( top10::math::Vector v)
{
  if (current_mesh == meshes.end()) return;
  top10::math::Translation4 T(v);
  top10::math::Matrix4 M = T * current_mesh->transform_node->toWorld();
  current_mesh->transform_node->setToWorld(M);
  notifyTransform();
}

void top10::helpers::MeshEditor::rotate( top10::math::AxisEnum ax, double angle_degrees )
{
  if (current_mesh == meshes.end()) return;
  
  top10::math::Vector v;
  switch (ax) {
    case top10::math::X: v = top10::math::Vector(1.0, 0.0, 0.0); break;
    case top10::math::Y: v = top10::math::Vector(0.0, 1.0, 0.0); break;
    case top10::math::Z: v = top10::math::Vector(0.0, 0.0, 1.0); break;
  }
  
  top10::math::Rotation3 R3(M_PI*angle_degrees/180.0, v);
  top10::math::Matrix4 R4(R3);
  R4 = R4*current_mesh->transform_node->toWorld();
  current_mesh->transform_node->setToWorld(R4);
  notifyTransform();
}

void
top10::helpers::MeshEditor::setFlag(MeshFlags flag)
{
  if (current_mesh == meshes.end()) return;

  current_mesh->flags |= flag;
  current_mesh->createNode();
}

void
top10::helpers::MeshEditor::clearFlag(MeshFlags flag)
{
  if (current_mesh == meshes.end()) return;

  current_mesh->flags &= ~flag;
  current_mesh->createNode();
}

std::string top10::helpers::MeshEditor::getMeshFilename( ) const
{
  if (current_mesh == meshes.end()) return "";
  return current_mesh->filename;
}

std::string top10::helpers::MeshEditor::getMeshName( ) const
{
  if (current_mesh == meshes.end()) return "";
  return current_mesh->name;
}

top10::math::Matrix4 top10::helpers::MeshEditor::getTransform( ) const
{
  if (current_mesh == meshes.end()) return top10::math::Matrix4();
  return current_mesh->transform_node->toWorld();
}

int top10::helpers::MeshEditor::getId( ) const
{
  if (current_mesh == meshes.end()) return -1;
  return current_mesh->id;
}

void top10::helpers::MeshEditor::clearState( )
{
  meshes.clear();
  current_mesh = meshes.begin();
  next_id = 0;
}

std::string top10::helpers::MeshEditor::getOrigin( ) const
{
  return "helpers::MeshEditor";
}

int top10::helpers::MeshEditor::loadXml( const TiXmlElement* xml_node)
{
  MeshList tmp;
  int next_id2=0;
  bool succeeded = true;
  
  for (const TiXmlElement* el = xml_node->FirstChildElement("mesh"); el; el = el->NextSiblingElement("mesh")) {
    int id2 = -1;
    int s = loadItem(el, &tmp, &id2);
    if (s < 0) succeeded = false;
    if (id2 >= next_id2) next_id2 = id2 +1;
  }
  
  meshes.swap(tmp);
  current_mesh = meshes.begin();
  next_id = next_id2;
  
  return succeeded?0:-1;
}

int top10::helpers::MeshEditor::saveXml( TiXmlElement* xml_node) const
{
  bool succeeded = true;
  for (MeshList::const_iterator it = meshes.begin(); it != meshes.end(); ++it)
  {
    TiXmlElement el("mesh");
    int s = saveItem(&el, it);
    if (s < 0) succeeded = false;
    xml_node->InsertEndChild(el);
  }
  
  return succeeded?0:-1;
}

int top10::helpers::MeshEditor::loadItem( const TiXmlElement* el, MeshList* m, int *id)
{
  MeshDesc new_one;
  new_one.flags = NO_FLAGS;
  int status = 0;
  
  const char* attr = el->Attribute("filename");
  if (attr) new_one.filename = attr;
  
  new_one.transform_node = new top10::graphX::TransformNode;
  const TiXmlElement* matrix_el = el->FirstChildElement("matrix");
  if (matrix_el) {
    top10::math::Matrix4 M;
    int s = top10::math::loadXml(matrix_el, &M);
    new_one.transform_node->setToWorld(M);
    if (s < 0) logXmlNodeError("Failed to load matrix", matrix_el);
  }
  
  int s = el->QueryIntAttribute("id", &new_one.id);
  if (s < 0) {
    logXmlNodeError("Failed to get id of a mesh", el, Log::Error);
    status = -1;
    new_one.id = -1;
  }
  else
    *id = new_one.id;
  
  attr = el->Attribute("name");
  if (!attr) {
    logXmlNodeError("Failed to get the name of an entry in list of meshes", el, Log::Error);
    status = -2;
  }
  else {
    std::string name(attr);
    top10::util::FindNamed finder(name);
    if (std::find_if(m->begin(), m->end(), finder) != m->end()) {
      logXmlNodeError("There is already an entry with name "+name, el, Log::Error);
      status = -3;
    }
    else
      new_one.name = name;
  }

  int value;
  s = el->QueryIntAttribute("show_front", &value);
  if (s == 0 && value != 0)
    new_one.flags |= FACE_SHOW_FRONT;

  s = el->QueryIntAttribute("show_back", &value);
  if (s == 0 && value != 0)
    new_one.flags |= FACE_SHOW_BACK;

  s = el->QueryIntAttribute("shadow_volume", &value);
  if (s == 0 && value != 0)
    new_one.flags |= SHADOW_VOLUME;

  s = el->QueryIntAttribute("no_collision", &value);
  if (s == 0 && value != 0)
    new_one.flags |= NO_COLLISION;

  if (status==0)
  {
    new_one.createNode();
    m->push_back(new_one);
  }

  return status;
}

int top10::helpers::MeshEditor::saveItem( TiXmlElement* el, MeshList::const_iterator it) const
{
  el->SetAttribute("filename", it->filename);
  el->SetAttribute("name", it->name);
  el->SetAttribute("id", it->id);
  if (it->flags & FACE_SHOW_FRONT)
    el->SetAttribute("show_front", 1);
  if (it->flags & FACE_SHOW_BACK)
    el->SetAttribute("show_back", 1);
  if (it->flags & SHADOW_VOLUME)
    el->SetAttribute("shadow_volume", 1);
  if (it->flags & NO_COLLISION)
    el->SetAttribute("no_collision", 1);

  TiXmlElement matrix_el("matrix");
  int s = top10::math::saveXml(&matrix_el, it->transform_node->toWorld());
  el->InsertEndChild(matrix_el);
  
  return s;
}

void top10::helpers::MeshEditor::MeshDesc::createNode( )
{
  try {
    top10::graphX::Node* top_node = 0;
    top10::graphX::Node* my_mesh_node = MeshFactory::getSingle()->load(filename);

    // Culling
    if (!(flags & FACE_SHOW_FRONT) != !(flags & FACE_SHOW_BACK))
    {
      top10::graphX::CullNode* cull_node = new top10::graphX::CullNode;
      cull_node->setCulling( !(flags & FACE_SHOW_FRONT) ? top10::graphX::RenderState::Front : top10::graphX::RenderState::Back );
      top_node = cull_node;
      cull_node->addChild(my_mesh_node);
    }
    else
    {
      top_node = my_mesh_node;
    }

    // Disable shadow volumes
    if (!(flags & SHADOW_VOLUME))
    {
      top10::graphX::LeafNodeIterator it(my_mesh_node);
      while (!it.atEnd())
      {
	top10::graphX::MeshNode* it_node = dynamic_cast<top10::graphX::MeshNode*>(*it);
	if (it_node)
	  it_node->setShadowVolumeFlag(false);
	++it;
      }
    }

    transform_node->removeChild(mesh_node.getPtr());
    mesh_node = top_node;
    transform_node->addChild(mesh_node.getPtr());
  }
  catch(const std::string& err) {
    Log::getSingle()->send(Log::Error, "MeshEditor::MeshDesc", err);
  }
}

void top10::helpers::MeshEditor::loadMesh( std::string filename )
{  
  if (current_mesh != meshes.end()) {
    current_mesh->filename = filename;
    current_mesh->createNode();
  }
}

top10::helpers::MeshEditor::MeshList::iterator top10::helpers::MeshEditor::find( std::string name )
{
  MeshList::iterator it;
  top10::util::FindNamed finder(name);
  return std::find_if(meshes.begin(), meshes.end(), finder);
}

top10::helpers::MeshEditor::MeshList::const_iterator top10::helpers::MeshEditor::find( std::string name ) const
{
  MeshList::const_iterator it;
  top10::util::FindNamed finder(name);
  return std::find_if(meshes.begin(), meshes.end(), finder);
}

std::string top10::helpers::MeshEditor::getMeshFilename( int id ) const
{
  MeshList::const_iterator it = find(id);
  if (it == meshes.end()) return "";
  else return it->filename;
}

std::string top10::helpers::MeshEditor::getMeshName( int id ) const
{
  MeshList::const_iterator it = find(id);
  if (it == meshes.end()) return "";
  else return it->name;
}

top10::math::Matrix4 top10::helpers::MeshEditor::getTransform( int id ) const
{
  MeshList::const_iterator it = find(id);
  if (it == meshes.end()) return top10::math::Matrix4();
  else return it->transform_node->toWorld();
}

int top10::helpers::MeshEditor::findId( std::string name ) const
{
  MeshList::const_iterator it = find(name);
  if (it == meshes.end()) return -1;
  else return it->id;
}

top10::helpers::MeshEditor::MeshList::iterator top10::helpers::MeshEditor::find( int id )
{
  top10::util::FindById finder(id);
  return std::find_if(meshes.begin(), meshes.end(), finder);
}

top10::helpers::MeshEditor::MeshList::const_iterator top10::helpers::MeshEditor::find( int id ) const
{
  top10::util::FindById finder(id);
  return std::find_if(meshes.begin(), meshes.end(), finder);
}

top10::graphX::Node* top10::helpers::MeshEditor::getNode( int id )
{
  MeshList::const_iterator it = find(id);
  if (it == meshes.end())
    return 0;
  else
    return it->transform_node.getPtr();
}

unsigned int top10::helpers::MeshEditor::getFlags(int id) const
{
  MeshList::const_iterator it = find(id);
  if (it == meshes.end())
    return 0;
  else
    return it->flags;
}

top10::graphX::Node* top10::helpers::MeshEditor::getCurrentNode( )
{
  if (current_mesh != meshes.end()) return current_mesh->transform_node.getPtr();
  return 0;
}

unsigned int
top10::helpers::MeshEditor::getFlags() const
{
  if (current_mesh != meshes.end())
    return current_mesh->flags;
  return 0;
}

void top10::helpers::MeshEditor::addListener(Listener* l)
{
  listeners.push_back(ListenerRef(l));
}

void top10::helpers::MeshEditor::removeListener(Listener* l)
{
  ListenerRef lref(l);
  ListenerRefs::iterator find_it = std::find(listeners.begin(), listeners.end(), lref);
  if (find_it != listeners.end())
    listeners.erase(find_it);
}

void top10::helpers::MeshEditor::Listener::notifyTransform(const top10::math::Matrix4&)
{
}

void top10::helpers::MeshEditor::notifyTransform() const
{
  for (int i=listeners.size()-1; i>=0; --i)
  {
    listeners[i]->notifyTransform(getTransform());
  }
}