/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_HELPERS_GENERIC_OCTREE_HH
#define TOP10_HELPERS_GENERIC_OCTREE_HH

#include "math/Vertex.hh"
#include "math/Box.hh"
#include <set>
#include <vector>
#include <list>

namespace top10 {
  namespace helpers {

    enum OverlapType { None=0, Intersect=1, Contains=2, Contained=3 };
    
    //! Function used when building a GenericOctree<ShapeT>
    /*! No default implementation. You must provide it using template specialization.
     minX, ..., maxZ must not be modified unless smaller (larger) values are found. */
    template<typename ShapeT>
    void GenericOctree__getBoundingBox(const ShapeT& shape,
				       double& minX, double& maxX,
				       double& minY, double& maxY,
				       double& minZ, double& maxZ);

    //! Function used to retrieve a part of the octree
    /*! No default implementation */
    template<typename VolumeT>
    OverlapType GenericOctree__overlap(const top10::math::AxisAlignedBox&, const VolumeT& client_volume);

    //! Function used when using a GenericOctree<ShapeT>
    /*! No default implementation. You must provide it using template specialization.
      Must do so for all VolumeTs used in GenericOctree::getBox */
    template<typename ShapeT>
    bool GenericOctree__intersect(const top10::math::AxisAlignedBox&, const ShapeT& shape);


    //! Class template for octrees of a certain type of shapes
    template<typename ShapeT>
    class GenericOctree {
      unsigned int POLY_THRESHOLD;
      int MAX_DEPTH;

    public:
      //! A vector of shapes
      typedef std::vector<ShapeT> ShapeVec;
      //! A set of indices inside the shapes vector
      typedef std::vector<int> ShapeRefs;

      //! Suitable for collision detection (low poly threshold)
      GenericOctree(const ShapeVec& shapes, int poly_threshold=5, int max_depth=20);
      GenericOctree();

      GenericOctree(const GenericOctree<ShapeT>& other);

      GenericOctree& operator=(const GenericOctree<ShapeT>& other);

      ~GenericOctree();

      //! Return a list of pointers to shapes that may intersect with a volume
      template<typename VolT>
      ShapeRefs getVolume(const VolT& vol, int max_depth=100) const;

      //! Return a pointer to the vector of shapes in this octree
      inline const ShapeVec* getShapeVec() const { return &shapes; }

      //! For debugging...
      std::list<top10::math::AxisAlignedBox> getAllBlocks() const;

      //! For debugging...
      template<typename VolumeT>
      std::list<top10::math::AxisAlignedBox> getBlocks(const VolumeT&, int max_depth=100) const;
      
    private:
      //! The structure of an internal node of an octree.
      struct Node {
	//! Bounding box for all shapes in this node
	top10::math::AxisAlignedBox box;
	//! Pointers to the eight children
	struct Node* children[8];
	//! Set of references to the shapes in this node
	ShapeRefs shape_refs;
      };

      /* Construction/deconstruction methods */
      //! Construct this octree
      void build();

      //! Destroy this octree
      void deconstruct();

      //! Duplicate a node and its children
      static Node* duplicateNode(const Node* node);
      
      //! Free a node and its children
      static void freeNode(Node* node);

      //! Build children nodes for a given node
      void splitNode(Node*, int depth);

      //! Constuct a node and its children
      Node* buildNode(top10::math::AxisAlignedBox box,
		      const ShapeRefs& polygons, int depth);


      /* Querying methods */

      //! Get the bounding boxes for a given node
      std::list<top10::math::AxisAlignedBox> getAllBlocks(const Node* node) const;

      //! Get the bounding boxes for a given node
      template<typename VolumeT>
      std::list<top10::math::AxisAlignedBox> getBlocks(const VolumeT&, const Node* node, int depth) const;

      //! Retrieve the vector of shape indices intersecting with a given volume
      template<typename VolT>
      static void getVolume(const VolT&, const Node*, int depth, ShapeRefs& refs, char* used_shapes);

      Node top_node;
      std::vector<ShapeT> shapes;
    };
  };
};

//#include "helpers/GenericOctree-template.cpp"

#endif
