/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include "3dsRead.hh"
#include "graphX/TextureManager.hh"

#include <lib3ds/file.h>                        
#include <lib3ds/mesh.h>
#include <lib3ds/node.h>
#include <lib3ds/material.h>
#include <lib3ds/matrix.h>
#include <lib3ds/vector.h>

#include <sstream>
#include <map>
#include <ctype.h>

using namespace top10;
using namespace top10::helpers;

using top10::util::Error;
using top10::math::Vector;
using top10::math::Matrix4;
using top10::math::Translation4;
using top10::math::Mesh;

Read3DS::Read3DS(std::string filename)
{
  path_finder = top10::util::PathFinder::defaultPathFinder();
  std::string filepath = path_finder.find(filename);
  if (filepath.empty()) throw Error(std::string("Could not find ")+filename);
  std::string path = top10::util::PathFinder::getPath(filepath);
  path_finder.addPath(path);
  path_finder.addPath(path + "/textures");
  
  Lib3dsFile* file = lib3ds_file_load(filepath.c_str());
  if (!file) throw Error(std::string("Could not open ")+filename);
  lib3ds_file_eval(file, 0);

  try {
    top_node = new top10::graphX::GroupNode;
    for (Lib3dsNode* node = file->nodes; node; node = node->next) {
      top_node->addChild(loadNode(file, node));
    }
  }
  catch (std::string err) {
    lib3ds_file_free(file);
    throw err;
  }
  lib3ds_file_free(file);
}

top10::graphX::Node* Read3DS::loadNode(Lib3dsFile* file, Lib3dsNode* node)
{
  using namespace top10::math;
  using namespace top10::graphX;
  
  GroupNode* group = new GroupNode;
    
  // Handle children if any
  for (Lib3dsNode* child = node->childs; child; child = child->next)
    group->addChild(loadNode(file, node));

  // Handle this mesh
  Lib3dsMesh* mesh=lib3ds_file_mesh_by_name(file, node->name);
  if (mesh == 0) return group;

  // Store vertices
  std::vector<Vector> points;
  for (int i=0; i<mesh->points; ++i) {
    Lib3dsPoint v = mesh->pointL[i];
    points.push_back(Vector(v.pos[0], v.pos[1], v.pos[2]));
  }

  // We will split one 3ds mesh into several Meshes, each with its own material
  
  // Mapping from names of materials to Meshes and their materials
  std::map<std::string, MeshMatTex*> material_to_mesh;
  
  // For each face...
  for (unsigned int p=0; p<mesh->faces; ++p) {
    Lib3dsFace *f=mesh->faceL + p;
    Lib3dsMaterial *mat=0;

    // Retrieve the material (surface)
    if (f->material[0]) {
      mat=lib3ds_file_material_by_name(file, f->material);
    }

    MeshMatTex* mmt = 0;
    if (mat) {
      // Check if there is already a Mesh for this material...
      std::map<std::string, MeshMatTex*>::const_iterator surf_f = material_to_mesh.find(std::string(f->material));

      // ... no
      if (surf_f == material_to_mesh.end()) {
        // Create the MaterialNode
	MaterialNode* my_mat = new MaterialNode;
	my_mat->r = (unsigned char)(mat->diffuse[0]*255);
	my_mat->g = (unsigned char)(mat->diffuse[1]*255);
	my_mat->b = (unsigned char)(mat->diffuse[2]*255);
 
        // Create the TextureNode
        TextureNode* my_texture = 0;
        
        if (mat->texture1_map.name[0]) {
          std::string nicename;
          for (const char*p = mat->texture1_map.name; *p; nicename += ::tolower(*p++));
          std::string texture_path = path_finder.find(nicename);
          if (!texture_path.empty()) {
            my_texture = new TextureNode;
            my_texture->setTextureId(top10::graphX::TextureManager::getInstance()->getTexture(texture_path));
          }
          else {
            std::cerr<<"Could not find texture file "<<nicename<<std::endl;
          }
        }
        
        // Create the Mesh
        Mesh* my_mesh = new Mesh;
        my_mesh->setVertices(points);
        
        mmt = new MeshMatTex;
        mmt->mesh = my_mesh;
        mmt->mat = my_mat;
        mmt->mat_name = f->material;
        mmt->texture = my_texture;
        material_to_mesh[std::string(f->material)] = mmt;
      }
      else mmt = surf_f->second;
    }
    else throw std::string("Missing material info");

    // Add this face
    mmt->mesh->addFace(f->points[0], f->points[1], f->points[2]);

    // Texels
    for (int i=0; i<3; ++i) {
      int idx = f->points[i];

      if (mesh->texels != 0) {
	if (mesh->texels != mesh->points) {
	  std::ostringstream buf;
	  buf<<"Bad number of texels "<<mesh->texels<<", should be: "<<mesh->points<<std::ends;
	  throw buf.str();
	}

	MeshNode::TextureCoord texel(mesh->texelL[idx][0], -mesh->texelL[idx][1]);
	mmt->tex_coords.push_back(texel);
      }
      else {
	top10::graphX::MeshNode::TextureCoord dummy;
	mmt->tex_coords.push_back(dummy);
      }
    }
  }

  // Magical transformations. 3DS manages transformation oddly, IMHO
  Lib3dsMatrix M;
  lib3ds_matrix_copy(M, mesh->matrix);
  lib3ds_matrix_inv(M);
  Matrix4 M2(M); M2 = M2.transpose();
  Matrix4 nodeM(node->matrix); nodeM = nodeM.transpose();
  Lib3dsObjectData* obj_data = &node->data.object;
  Matrix4 nodeT = Translation4(-Vector(obj_data->pivot)); /*nodeT = nodeT.transpose();*/
  Matrix4 to_world = nodeM*nodeT*M2;
  
  try {
    // Top node for all the split meshes
    TransformNode* trans_node = new TransformNode;
    trans_node->setToWorld(to_world);
    // Set the name of the top node, so that we can retrieve individual nodes in the scene graph by their name
    trans_node->setName(node->name);
    
    // Add all children: one line for each material
    for (std::map<std::string, MeshMatTex*>::const_iterator it = material_to_mesh.begin(); it != material_to_mesh.end(); ++it) {
      if (it->second->texture) {
        trans_node->addChild(it->second->texture);
        it->second->texture->addChild(it->second->mat);
      } else {
        trans_node->addChild(it->second->mat);
      }
      it->second->mesh->canonizeFaces();
      MeshNode* mesh_node = new top10::graphX::MeshNode(it->second->mesh, it->second->tex_coords, node->name);
      mesh_node->setMaterialName(it->first);
      it->second->mat->addChild(mesh_node);
    
    }
    
    group->addChild(trans_node);
  }
  catch (NonInvertible<4>) {}
  
  return group;
}
