/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include <SDL.h>
//#include <GL/glu.h>

#include "util/PathFinder.hh"
#include "helpers/3dsRead.hh"
#include "helpers/ReadCal3d.hh"
#include "helpers/initGL.hh"
#include "FreeCameraNode.hh"
#include "GroupNode.hh"
#include "LightNode.hh"
#include "Renderer.hh"
#include "GridNode.hh"
#include "SkyBoxNode.hh"
#include "ShadowVolumeNode.hh"
#include "CullNode.hh"
#include "PanelNode.hh"
#include "SwitchNode.hh"
#include "TransformNode.hh"

using namespace top10::graphX;


int main(int argc, char** argv)
{
  try {
    int width = 800;
    int height = 600;
    std::string filename;
    
    // Parse comand line options
    for (int i=1; i<argc; ++i) {
      if (std::string(argv[i]) == "-w") {
	++i;
	if (i >= argc) throw std::string("Missing value: width");
	width = atoi(argv[i]);
      }
      else if (std::string(argv[i]) == "-h") {
	++i;
	if (i >= argc) throw std::string("Missing value: height");
	height = atoi(argv[i]);
      }
      else {
	filename = argv[i];
      }
    }
    if (filename.empty()) throw std::string("Missing filename");      
    
    // SDL init
    if ( SDL_Init(SDL_INIT_VIDEO) < 0 )
      throw std::string("Couldn't initialize SDL: ") + std::string(SDL_GetError());
    SDL_Surface* screen = top10::helpers::initGL(width, height, false /* No fullscreen */);

    // GL init
    Renderer render;
    render.initGL(width, height);
    
    // Find the location of the model file
    top10::util::PathFinder path_finder = top10::util::PathFinder::defaultPathFinder();
    std::string path = path_finder.find(filename);
    if (path.empty()) {
      throw std::string("Could not find ")+filename;
    }
    
    // Load the file
    Node* model = 0;
    try {
      top10::helpers::Read3DS model3ds(path.c_str());
      model = model3ds.getTopNode();
    }
    catch(...) {
      top10::helpers::ReadCal3d model3ds(path.c_str());
      model = model3ds.getTopNode();
    }

    unsigned int quality_n = 0;
    const unsigned int N_QUALITIES=5;
    const char* qualities[N_QUALITIES] = {"VeryHigh", "High", "Medium", "Low", "VeryLow"};
    ShadowVolumeNode::Method shadow_method = ShadowVolumeNode::toMethod(qualities[quality_n]);

    // Create a shadow volume
    Node* shadow_volumes = makeProxy(model,
				     new ShadowVolumeNode::MyProxyOperation(shadow_method),
				     true);
    
    // Create a ground
    TransformNode* ground_transform = new TransformNode;
    top10::math::Identity4 M;
    M(1,1) = 0.0;
    M(2,1) = -1.0;
    M(2,2) = 0.0;
    M(1,2) = 1.0;
    ground_transform->setToWorld(M);

    PanelNode* ground = new PanelNode;
    ground->setLowerLeft(-100.0, -100.0);
    ground->setUpperRight(100.0, 100.0);
    ground->setDist(-0.1);

    MaterialNode* gray = new MaterialNode;
    gray->r = gray->g = gray->b = 200;
    gray->addChild(ground);

    ground_transform->addChild(gray);

    SwitchNode* switch_ground = new SwitchNode;
    switch_ground->addChild(ground_transform);

    int show_ground = 1;

    // Create a camera    
    FreeCameraNode* camera = new FreeCameraNode;
    FreeCameraNode* camera1 = new FreeCameraNode;
    FreeCameraNode* camera2 = new FreeCameraNode;
    FreeCameraNode* camera3 = new FreeCameraNode;
    FreeCameraNode* camera4 = new FreeCameraNode;
    render.setCamera(camera);
    
    // Create a light
    LightNode* light = new LightNode;
    ShadowVolumeNode::recurseUpdate(light->getLight().getPosition(), shadow_volumes);
    
    // Create a glowing grid representing the ground
    GridNode* grid = new GridNode;
    MaterialNode* glowing = new MaterialNode;
    glowing->b = 255;
    glowing->glow_b = 255;
    glowing->addChild(grid);
    
    // To show the shadow volume
    MaterialNode* glowing2 = new MaterialNode;
    glowing2->b = 255;
    glowing2->glow_b = 255;
    glowing2->addChild(shadow_volumes);
    
    // Put everyone into one graph
    CullNode* top = new CullNode;
    top->setCulling(RenderState::Back);
    top->addChild(light);
    top->addChild(camera);
    top->addChild(glowing);
    light->addChild(model);
    light->addChild(switch_ground);

    // Create a skybox
    SkyBoxNode* skybox = new SkyBoxNode;
        
    SDL_Event event;
    bool show_shadow_volume = false;
    bool show_shadows = false;
    RenderList rl;
    
    for (bool leave = false; !leave;) {
      // Draw the scene
      render.clearAllGL();
      
      rl.clear();
      render.buildList(skybox, &rl);
      render.renderSkyBoxGL(rl);
      glClear(GL_DEPTH_BUFFER_BIT);
      
      rl.clear();
      render.buildList(top, &rl);
      render.renderGL(rl);

      if (show_shadow_volume) {
	rl.clear();
	render.buildList(glowing2, &rl);
        render.renderGL(rl);
      }
      
      if (show_shadows) {
        ShadowVolumeNode::renderShadows(&render, shadow_volumes, 100);
      }
      
      SDL_GL_SwapBuffers();
      
      // Handle events
      while (SDL_PollEvent(&event)) {
	switch (event.type) {
      	case SDL_KEYDOWN:
	  if (event.key.keysym.sym == SDLK_ESCAPE) leave=true;
          else if (event.key.keysym.sym == SDLK_SPACE) {
            Light new_light = light->getLight();
            new_light.setPosition(camera->getView().getCenter());
            new_light.setDirection(camera->getView().getDirection());
            light->setLight(new_light);
            ShadowVolumeNode::recurseUpdate(camera->getView().getCenter(), shadow_volumes);
          }
          else if (event.key.keysym.sym == SDLK_F1) {
            *camera1 = *camera;
	    std::cout << "Saved camera position 1" << std::endl;
          }
          else if (event.key.keysym.sym == SDLK_F2) {
            *camera2 = *camera;
	    std::cout << "Saved camera position 2" << std::endl;
          }
          else if (event.key.keysym.sym == SDLK_F3) {
            *camera3 = *camera;
	    std::cout << "Saved camera position 3" << std::endl;
          }
          else if (event.key.keysym.sym == SDLK_F4) {
            *camera4 = *camera;
	    std::cout << "Saved camera position 4" << std::endl;
          }
          
          else if (event.key.keysym.sym == SDLK_F5) {
            *camera = *camera1;
	    std::cout << "Restored camera position 1" << std::endl;
          }
          else if (event.key.keysym.sym == SDLK_F6) {
            *camera = *camera2;
	    std::cout << "Restored camera position 2" << std::endl;
          }
          else if (event.key.keysym.sym == SDLK_F7) {
            *camera = *camera3;
	    std::cout << "Restored camera position 3" << std::endl;
          }
          else if (event.key.keysym.sym == SDLK_F8) {
            *camera = *camera4;
	    std::cout << "Restored camera position 4" << std::endl;
          }
          else if (event.key.keysym.sym == SDLK_s) {
            show_shadows = !show_shadows;
	    std::cout << "Show shadows is now: " << show_shadows << std::endl;
          }
          else if (event.key.keysym.sym == SDLK_v) {
            show_shadow_volume = !show_shadow_volume;
	    std::cout << "Show shadow volume is now: " << show_shadow_volume << std::endl;
          }
          else if (event.key.keysym.sym == SDLK_q) {
	    ++quality_n;
	    quality_n %= N_QUALITIES;
	    shadow_method = ShadowVolumeNode::toMethod(qualities[quality_n]);
	    glowing2->removeChild(shadow_volumes);
	    shadow_volumes = makeProxy(model,
				       new ShadowVolumeNode::MyProxyOperation(shadow_method),
				       true);
	    glowing2->addChild(shadow_volumes);

            ShadowVolumeNode::recurseUpdate(light->getLight().getPosition(), shadow_volumes);

	    std::cout << "Quality is now: " << qualities[quality_n] << std::endl;
	  }
          else if (event.key.keysym.sym == SDLK_g) {
            show_ground = 1-show_ground;
	    switch_ground->enable(1-show_ground);
	    std::cout << "Show ground is now: " << show_ground << std::endl;
          }

	  break;

      	case SDL_QUIT:
      	  leave = true;
          break;

      	case SDL_MOUSEMOTION:
	  if (event.motion.state == 1) {
            camera->rotateY(90.0 * event.motion.xrel / height);
            camera->rotateX(90.0 * event.motion.yrel / width);
	  }
	  if (event.motion.state > 1) {
	    camera->translateL(top10::math::Vector(-50.0*event.motion.xrel/width, 0.0, 50.0*event.motion.yrel/height));
	  }
          camera->update();

	  break;
      		
      	default:
      	  break;
      	}
      }
    }
  }
  catch (std::string err) {
    std::cerr<<err<<std::endl;
    return 1;
  }
  
  return 0;
}
