/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/
#include "Wireframe.hh"
#include "CameraNode.hh"

namespace top10 {
namespace graphX {

Wireframe::Wireframe(): thickness(1.0), my_op(new MyProxyOperation) {}

void Wireframe::modifyRenderState(RenderState& s, const CameraNode&) const
{
  s.setLineWidth(thickness);
  s.setTextureId(0);
  s.disableLights();
  s.disableChanges(RenderState::MaterialBit);
  s.disableChanges(RenderState::TextureBit);
  s.disableChanges(RenderState::LightsBit);
}

Node* Wireframe::getChild(int idx) const
{
  if (idx < 0 || idx >= children.size()) return 0;
  return children[idx].getPtr();
}

void Wireframe::addChild(Node* n)
{
  MeshNode* mesh_node = dynamic_cast<MeshNode*>(n);
  TransformNode* trans = dynamic_cast<TransformNode*>(n);
  if (mesh_node) children.push_back(NodeRef(new WireframeMesh(mesh_node)));
  else if (trans) children.push_back(NodeRef(new TransformNodeProxy(trans, my_op.getPtr(), true)));
  else children.push_back(NodeRef(new NodeProxy(n, my_op.getPtr(), true)));
  
  originals.push_back(NodeRef(n));
}

void Wireframe::removeChild(Node* n)
{
  assert(originals.size() == children.size());
  for (int i=0; i<originals.size(); ++i) {
    if (originals[i].getPtr() == n) {
      originals.erase(originals.begin() + i);
      children.erase(children.begin() + i);
      return;
    }
  }
}

void Wireframe::renderGL(const RenderingFeatures&, const RenderState&, const CameraNode&) const {}

Node* Wireframe::MyProxyOperation::makeChild(const MeshNode* m) const
{
  return new WireframeMesh(m);
}

void Wireframe::WireframeMesh::renderGL(const RenderingFeatures&, const RenderState& rs, const CameraNode& cam) const
{
  using top10::math::Mesh;
  using top10::math::Vector;

  // Transformation to apply to normals: ignore the translation
  top10::math::Matrix4 T = rs.getTransform();
  T(0,3) = T(1,3) = T(2,3) = 0;
  
  const std::vector<Mesh::Face>* faces = orig->getMesh()->getFaces();
  const std::vector<Vector>* vertices = orig->getMesh()->getVertices();

  std::vector<int>::const_iterator normal_idx_it = orig->getNormalIndices()->begin();
  for (std::vector<Mesh::Face>::const_iterator face_it = faces->begin();
       face_it != faces->end();
       ++face_it) {
    // Check if we must cull this face
    if (rs.getCulling() != RenderState::NoCulling) {
      Vector n = orig->getNormals()->at(*normal_idx_it++)
        + orig->getNormals()->at(*normal_idx_it++)
        + orig->getNormals()->at(*normal_idx_it++);
      n = T * n;
      double s = n*cam.getView().getDirection();
      
      if (rs.getCulling() == RenderState::Front && s < 0) continue;
      if (rs.getCulling() == RenderState::Back && s > 0) continue;
    }
    else normal_idx_it += 3;
    
    glBegin(GL_LINE_LOOP);
    for (int i=0; i<3; ++i) {
      top10::math::Vector v = vertices->at(face_it->idxs[i]);
      glVertex3d(v.x, v.y, v.z);
    }
    glEnd();    
  }

}

void WireframeFill::modifyRenderState(RenderState& s, const CameraNode&) const
{
  s.setPolyFill(RenderState::Lines);
  s.setLineWidth(thickness);
  s.setTextureId(0);
  s.disableLights();
  s.disableChanges(RenderState::MaterialBit);
  s.disableChanges(RenderState::TextureBit);
  s.disableChanges(RenderState::LightsBit);
}

}
}
