/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_GRAPHX_TRANSFORMNODE_HH
#define TOP10_GRAPHX_TRANSFORMNODE_HH

#include "GroupNode.hh"

namespace top10 {
  namespace graphX {
    class TransformNode: public GroupNode {
    public:
      TransformNode(): to_world(top10::math::Identity4()),
		       from_world(top10::math::Identity4())
	{}

      //! Return the transformation matrix from this node's coordinate system to the world's
      inline const top10::math::Matrix4& toWorld() const { return to_world; }

      //! Return the transformation matrix from the world's coordinate system to this node's
      inline const top10::math::Matrix4& fromWorld() const { return from_world; }

      //! Set the transformation matrix from this node's coordinate system to the world's
      /*! Also updates the FromWorld matrix */
      inline void setToWorld(const top10::math::Matrix4& M) { from_world = inverse(M); to_world = M; }

      //! Set the transformation matrix from the world's coordinate system to this node's
      /*! Also updates the ToWorld matrix */
      inline void setFromWorld(const top10::math::Matrix4& M) { to_world = inverse(M); from_world = M; }

      //! Set the transformation matrix and its inverse at once
      inline void setTransform(const top10::math::Matrix4& to, const top10::math::Matrix4& from) {
	to_world = to; from_world = from;
      }

    protected:
      void modifyRenderState(RenderState& s, const CameraNode&) const {
	s.setTransform(s.getTransform()*to_world);
      }

    private:
      top10::math::Matrix4 to_world;
      top10::math::Matrix4 from_world;
    };

    typedef top10::util::Ref< TransformNode > TransformNodeRef;
    typedef std::vector< TransformNodeRef > TransformNodeRefs;
  };
};

#endif
