/*
  Top10, a racing simulator
  Copyright (C) 2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "SphereNode.hh"

#include <GL/glu.h>

#include "math/Vertex.hh"
#include "util/Log.hh"

void __stdcall error_cb(GLenum err)
{
  using top10::util::Log;
  Log::getSingle()->send(Log::Error, "SphereNode", "GLU quadric error: " + std::string(reinterpret_cast<const char*>(gluErrorString(err))));
}

namespace top10
{
  namespace graphX
  {
    /* Structure contening private data of SphereNode.
       Having it defined here instead of the header makes it possible to isolate the header from OpenGL.
       Not really required yet, just experimenting with the idea...
       */
    struct SphereNodePrivate
    {
      GLUquadricObj* quadric;
      GLuint disp_list;
    };

    SphereNode::SphereNode(bool smooth)
    {
      data = new SphereNodePrivate;

      data->quadric = gluNewQuadric();
      gluQuadricCallback(data->quadric, GLU_ERROR, (void(__stdcall*)())&error_cb);

      gluQuadricDrawStyle(data->quadric, GLU_FILL);
      gluQuadricOrientation(data->quadric, GLU_OUTSIDE);
      gluQuadricNormals(data->quadric, smooth? GLU_SMOOTH : GLU_FLAT);
      data->disp_list = glGenLists(1);
      glNewList(data->disp_list, GL_COMPILE);
      gluSphere(data->quadric, 1.0, n_slices, n_stacks);      
      glEndList();
    }

    SphereNode::~SphereNode()
    {
      glDeleteLists(data->disp_list, 1);
      gluDeleteQuadric(data->quadric);
      delete data;
    }

    void SphereNode::renderGL(const RenderingFeatures&, const RenderState&, const CameraNode&) const
    {
      glCallList(data->disp_list);
    }
  }
}