/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#ifndef TOP10_GRAPHXSHADOWVOLUMENODE_HH
#define TOP10_GRAPHXSHADOWVOLUMENODE_HH

#include "LeafNode.hh"
#include "NodeProxy.hh"
#include "math/Outline.hh"
#include "util/NoCopy.hh"

namespace top10 {
namespace graphX {

class Renderer;
class ShadowVolumeMaker;

class ShadowVolumeNode : public LeafNode, public top10::util::NoCopy
{
public:
  enum Method {
    DynamicOutline,
    StaticOutlineHigh,
    StaticOutlineMedium,
    StaticOutlineLow,
    StaticBox
  };

  static Method toMethod(const std::string&);

  class MyProxyOperation: public ProxyOperation
  {
  public:
    MyProxyOperation(ShadowVolumeNode::Method);
    Node* makeChild(const MeshNode*) const;
  private:
    ShadowVolumeNode::Method m_method;
  };
  

public:
  ShadowVolumeNode(const top10::math::Mesh*, Method);
  ~ShadowVolumeNode();

  inline const top10::math::Mesh* getMesh() const { return m_mesh; }

  void update(top10::math::Vector light_pos);

  void renderGL(const RenderingFeatures&, const RenderState&, const CameraNode&) const;
  
  static void recurseUpdate(top10::math::Vector light_pos, Node*, top10::math::Matrix4 M);
  static void recurseUpdate(top10::math::Vector light_pos, Node*);

  //! Helper function to render shadows
  static void renderShadows(Renderer*, Node* shadow_node, unsigned char alpha);
    
  
public: // Interface for ShadowVolumeMaker
  void clearArrays();

  void project(const top10::math::Vector& p1, const top10::math::Vector& p2,
	       const top10::math::Vector& light_pos);

  template< typename IteratorT >
  void project(IteratorT begin, IteratorT end,
	       const top10::math::Vector& light_pos);

  void addQuad(const top10::math::Vector&,
	       const top10::math::Vector&,
	       const top10::math::Vector&,
	       const top10::math::Vector&);

private:
  void pushVertex(const top10::math::Vector& p);
  void pushNormal(const top10::math::Vector& p);

  const top10::math::Mesh* m_mesh;
  ShadowVolumeMaker* m_volume_maker;
  std::vector<float> m_vertex_array;
  std::vector<float> m_normal_array;
};



//! Interface to build a volume given a light position.
class ShadowVolumeMaker: public top10::util::NoCopy
{
public:
  ShadowVolumeMaker(ShadowVolumeNode*);
  virtual ~ShadowVolumeMaker();
  virtual void buildVolume(top10::math::Vector light_pos) =0;

protected:
  inline ShadowVolumeNode* getVol() const { return m_volume; }

private:
  ShadowVolumeNode* m_volume;
};



//! Nicest and slowest.
/*!
  Computes a new outline at every update.
*/
class SVM_DynamicOutline: public ShadowVolumeMaker
{
public:
  SVM_DynamicOutline(ShadowVolumeNode*);
  void buildVolume(top10::math::Vector light_pos);
};



//! Average.
/*!
  Computes three outlines at construction time.
*/
class SVM_StaticOutline: public ShadowVolumeMaker
{
public:
  SVM_StaticOutline(ShadowVolumeNode*, double min_dist);
  void buildVolume(top10::math::Vector light_pos);

private:
  std::list<top10::math::Vector> m_side;
  std::list<top10::math::Vector> m_top;
  std::list<top10::math::Vector> m_front;
};



//! Fast.
/*!
  Use a horizontal rectangle as the outline.
*/
class SVM_Basic: public ShadowVolumeMaker
{
public:
  SVM_Basic(ShadowVolumeNode*);
  void buildVolume(top10::math::Vector light_pos);

private:
  std::vector<top10::math::Vector> m_shape;
};



}
}

#endif
