/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/
#include "ShadowVolumeNode.hh"
#include "Renderer.hh"
#include "CullNode.hh"
#include "PanelNode.hh"
#include "AlphaNode.hh"
#include "MaterialNode.hh"

namespace top10 {
namespace graphX {

ShadowVolumeNode::ShadowVolumeNode(const top10::math::Mesh* mesh, Method method):
  m_mesh(mesh)
{
  switch (method)
  {
  case DynamicOutline:      m_volume_maker = new SVM_DynamicOutline(this); break;
  case StaticOutlineHigh:   m_volume_maker = new SVM_StaticOutline(this, 0.00); break;
  case StaticOutlineMedium: m_volume_maker = new SVM_StaticOutline(this, 0.03); break;
  case StaticOutlineLow:    m_volume_maker = new SVM_StaticOutline(this, 0.10); break;
  case StaticBox:           m_volume_maker = new SVM_Basic(this); break;
  }
}

ShadowVolumeNode::~ShadowVolumeNode()
{
  delete m_volume_maker;
}

void ShadowVolumeNode::pushVertex(const top10::math::Vector& p)
{
  for (int i=0; i<3; ++i)
    m_vertex_array.push_back(p[i]);  
}

void ShadowVolumeNode::pushNormal(const top10::math::Vector& p)
{
  for (int i=0; i<3; ++i)
    m_normal_array.push_back(p[i]);  
}

void ShadowVolumeNode::addQuad(const top10::math::Vector& p1,
			       const top10::math::Vector& p2,
			       const top10::math::Vector& p3,
			       const top10::math::Vector& p4)
{
  pushVertex(p1);
  pushVertex(p2);
  pushVertex(p3);
  pushVertex(p4);

  pushNormal((p2-p1) ^ (p3-p2));
}

void ShadowVolumeNode::project(const top10::math::Vector& p1, const top10::math::Vector& p2,
			       const top10::math::Vector& light_pos)
{
    top10::math::Vector v[4];

    v[0] = p1;
    v[1] = p2;
    v[2] = v[1] + 10.0 * (v[1] - light_pos);
    v[3] = v[0] + 10.0 * (v[0] - light_pos);
    top10::math::Vector n = (v[1] - v[0]) ^ (v[2] - v[1]);  // No need to normalize, we only care about the orientation

    for (int j=0; j<4; ++j)
      for (int i=0; i<3; ++i) {
        m_vertex_array.push_back(v[j][i]);
        m_normal_array.push_back(n[i]);
      }

}

template<typename IteratorT>
void ShadowVolumeNode::project(IteratorT begin, IteratorT end,
			       const top10::math::Vector& light_pos)
{
  IteratorT it, it2;
  it = begin;
  it2 = it;
  
  if (it2 != end)
    ++it2;
  else
    return;

  if (it2 == end)
    return;

  while (it2 != end)
  {
    project(*it, *it2, light_pos);

    it = it2;
    ++it2;
  }
  it2 = begin;
  project(*it, *it2, light_pos);
}

void ShadowVolumeNode::update(top10::math::Vector light_pos)
{
  m_volume_maker->buildVolume(light_pos);
}



void ShadowVolumeNode::clearArrays()
{
  m_vertex_array.erase(m_vertex_array.begin(), m_vertex_array.end());
  m_normal_array.erase(m_normal_array.begin(), m_normal_array.end());
}



void ShadowVolumeNode::renderGL(const RenderingFeatures& features, const RenderState& s, const CameraNode& camera) const
{
  if (m_vertex_array.empty())
    return;

  glEnableClientState(GL_VERTEX_ARRAY);
  glEnableClientState(GL_NORMAL_ARRAY);
  glVertexPointer(3, GL_FLOAT, 0, (const GLvoid*)&(m_vertex_array[0]));
  glNormalPointer(GL_FLOAT, 0, (const GLvoid*)&(m_normal_array[0]));
  glDrawArrays(GL_QUADS, 0, m_vertex_array.size()/3);
  glDisableClientState(GL_NORMAL_ARRAY);
  glDisableClientState(GL_VERTEX_ARRAY);  
}

void ShadowVolumeNode::recurseUpdate(top10::math::Vector light_pos, Node* n)
{
  top10::math::Matrix4 I = top10::math::Identity4();
  recurseUpdate(light_pos, n, I);   
}
      
void ShadowVolumeNode::recurseUpdate(top10::math::Vector light_pos, Node* n, top10::math::Matrix4 M)
{
  ShadowVolumeNode* vn = dynamic_cast<ShadowVolumeNode*>(n);
  if (vn) {
    M = inverse(M);
    vn->update(M*light_pos);
  }
  else {
    TransformNode* tn = dynamic_cast<TransformNode*>(n);
    if (tn) M = M*tn->toWorld(); 
  
    else {
      TransformNodeProxy* tn2 = dynamic_cast<TransformNodeProxy*>(n);
      if (tn2) M = M*tn2->toWorld();
    }
  }
  
  Node* child;
  int i=0;
  while ((child=n->getChild(i++))) recurseUpdate(light_pos, child, M);
}
  
ShadowVolumeNode::MyProxyOperation::MyProxyOperation(ShadowVolumeNode::Method method):
  m_method(method)
{
}

Node* ShadowVolumeNode::MyProxyOperation::makeChild(const MeshNode* m) const
{
  if (!m->getShadowVolumeFlag())
    return new GroupNode; // Return an empty group node.

  return new ShadowVolumeNode(m->getMesh(), m_method);
}

void ShadowVolumeNode::renderShadows(Renderer* render, Node* shadows, unsigned char shadows_alpha)
{
  // Create a culling node
  top10::util::Ref<top10::graphX::CullNode> cull_front(new top10::graphX::CullNode);
  top10::util::Ref<top10::graphX::CullNode> cull_back(new top10::graphX::CullNode);
  
  // Cull back faces of the shadow volumes
  cull_back->setCulling(top10::graphX::RenderState::Back);
  cull_back->addChild(shadows);
  // Cull front faces of the shadow volumes
  cull_front->setCulling(top10::graphX::RenderState::Front);
  cull_front->addChild(shadows);
  
  // enable stencil test, disable rendering
  render->toggleColorRendering(false);
  render->toggleDepthRendering(false);
  render->enableStencil(GL_ALWAYS, 0);
  
  // Render front faces into the stencil buffer
  RenderList rl;
  render->buildList(cull_back.getPtr(), &rl);
  render->setStencilOp(GL_KEEP, GL_KEEP, GL_INCR);
  render->renderGL(rl);
  
  // Render back faces into the stencil buffer
  rl.clear();
  render->buildList(cull_front.getPtr(), &rl);
  render->setStencilOp(GL_KEEP, GL_KEEP, GL_DECR);  
  render->renderGL(rl);

  // Render the "shadows": a transparent 2d black panel
  top10::util::Ref<top10::graphX::PanelNode> panel(new top10::graphX::PanelNode);
  top10::util::Ref<top10::graphX::AlphaNode> alpha(new top10::graphX::AlphaNode);
  alpha->setAlpha(shadows_alpha);
  alpha->addChild(panel.getPtr());
  panel->setDist(0);

#if 0
  //alpha->setAlpha(255);
  // Debug: show the value of the three lower bits of the stencil buffer using colors
  top10::util::Ref<MaterialNode> red(new MaterialNode);
  top10::util::Ref<MaterialNode> green(new MaterialNode);
  top10::util::Ref<MaterialNode> blue(new MaterialNode);
  red->r = 255;
  green->g = 255;
  blue->b = 255;
  red->addChild(alpha.getPtr());
  green->addChild(alpha.getPtr());
  blue->addChild(alpha.getPtr());
  
  render->toggleColorRendering(true);
  render->toggleDepthTest(false);
  render->setStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);
  
  render->enableStencil(GL_EQUAL, 1, 1);
  rl.clear();
  render->buildList(blue.getPtr(), &rl);
  render->renderHudGL(rl);
  
  render->enableStencil(GL_EQUAL, 2, 2);
  rl.clear();
  render->buildList(green.getPtr(), &rl);
  render->renderHudGL(rl);
  
  render->enableStencil(GL_EQUAL, 4, 4);
  rl.clear();
  render->buildList(red.getPtr(), &rl);
  render->renderHudGL(rl);
#else
  render->toggleColorRendering(true);
  render->enableStencil(GL_NOTEQUAL, (1<<render->getStencilBits())/2);
  rl.clear();
  render->buildList(alpha.getPtr(), &rl);
  render->toggleDepthTest(false);
  render->renderHudGL(rl);
#endif
  
  // Restore the rendering flags
  render->toggleDepthRendering(true);
  render->toggleDepthTest(true);
  render->disableStencil();  
}


ShadowVolumeNode::Method
ShadowVolumeNode::toMethod(const std::string& str)
{
  if (str == "Volume" || str == "DynamicOutline" || str == "VeryHigh")
  {
    return DynamicOutline;
  }
  else if (str == "StaticOutline" || str == "High")
  {
    return StaticOutlineHigh;
  }
  else if (str == "StaticOutline" || str == "Medium")
  {
    return StaticOutlineMedium;
  }
  else if (str == "Low")
  {
    return StaticOutlineLow;
  }
  else if (str == "StaticBox" || str == "VeryLow")
  {
    return StaticBox;
  }
  return DynamicOutline;
}

/*
 * ShadowVolumeMaker
 */

ShadowVolumeMaker::ShadowVolumeMaker(ShadowVolumeNode* vol):
  m_volume(vol)
{
}



ShadowVolumeMaker::~ShadowVolumeMaker()
{
}



/*
 * SVM_DynamicOutline
 */
SVM_DynamicOutline::SVM_DynamicOutline(ShadowVolumeNode* vol):
  ShadowVolumeMaker(vol)
{
}



void SVM_DynamicOutline::buildVolume(top10::math::Vector light_pos)
{
  using top10::math::Outline;

  getVol()->clearArrays();

  std::list<Outline::Edge> edges;
  top10::math::Outline outline(getVol()->getMesh());
  edges = outline.compute(light_pos);
  
  for (std::list<Outline::Edge>::const_iterator it = edges.begin(); it != edges.end(); ++it)
  {
    getVol()->project(it->p1, it->p2, light_pos);
  }
}



/*
 * SVM_StaticOutline
 */
SVM_StaticOutline::SVM_StaticOutline(ShadowVolumeNode* vol, double min_dist):
  ShadowVolumeMaker(vol)
{
  const double min_dist2 = min_dist * min_dist;
  using top10::math::Outline;

  Outline outline(getVol()->getMesh());
  std::list<top10::math::Outline::Edge> edges;

  edges = outline.compute( top10::math::Vector(50.0, 0.0, 0.0) );
  m_side = Outline::sort(edges);
  Outline::simplify(&m_side, min_dist2);

  edges = outline.compute( top10::math::Vector(0.0, 50.0, 0.0) );
  m_top = Outline::sort(edges);
  Outline::simplify(&m_top, min_dist2);

  edges = outline.compute( top10::math::Vector(0.0, 0.0, 50.0) );
  m_front = Outline::sort(edges);
  Outline::simplify(&m_front, min_dist2);
}



void SVM_StaticOutline::buildVolume(top10::math::Vector light_pos)
{
  getVol()->clearArrays();

  getVol()->project(m_side.begin(), m_side.end(), light_pos);
  getVol()->project(m_top.begin(), m_top.end(), light_pos);
  getVol()->project(m_front.begin(), m_front.end(), light_pos);
}



/*
 * SVM_Basic
 */
SVM_Basic::SVM_Basic(ShadowVolumeNode* vol):
  ShadowVolumeMaker(vol)
{
  using top10::math::Vector;
  Vector v_min = getVol()->getMesh()->getBoundMin();
  Vector v_max = getVol()->getMesh()->getBoundMax();

  m_shape.reserve(4);
  m_shape.push_back(Vector(v_max.x, v_max.y, v_min.z));
  m_shape.push_back(Vector(v_max.x, v_min.y, v_min.z));
  m_shape.push_back(Vector(v_min.x, v_min.y, v_min.z));
  m_shape.push_back(Vector(v_min.x, v_max.y, v_min.z));
}



void SVM_Basic::buildVolume(top10::math::Vector light_pos)
{
  getVol()->clearArrays();
  getVol()->project(m_shape.begin(), m_shape.end(), light_pos);
  getVol()->addQuad(m_shape[0], m_shape[1], m_shape[2], m_shape[3]);
}

}
}
