/*
  Top 10, a racing simulator
  Copyright (C) 2003-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#ifndef TOP10_GRAPHXRENDERER_HH
#define TOP10_GRAPHXRENDERER_HH

#include "RenderList.hh"
#include "RenderingFeatures.hh"

namespace top10 {
namespace graphX {

class CameraNode;

class Renderer {
public:
  Renderer();

  //! Set the current camera
  void setCamera(CameraNode*);
  CameraNode* getCamera() const;
  
  //! Build a sorted list of nodes.
  /*! The list is sorted to minimize OpenGL state changes. */
  void buildList(const Node*, RenderList* out_rl) const;
  //! Render the list, ignoring the position of the camera, but not the orientation.
  inline void renderSkyBoxGL(const RenderList& rl) { renderCommon(Skybox, rl); }
  //! Render the list
  inline void renderGL(const RenderList& rl) { renderCommon(Scene, rl); }
  //! Render a 2d HUD
  inline void renderHudGL(const RenderList& rl) { renderCommon(Hud, rl); }
  
  //! Initialize the OpenGL state.
  /*! \note must be called before rendering */
  void initGL(int width, int height);

  //! Change the viewport
  void resizeGL(int width, int height);

  //! Clear all OpenGL buffers
  void clearAllGL();
  //! Clear the OpenGL depth buffer
  void clearDepthGL();
  //! Clear the color buffer
  void clearColorGL();
  //! Clear the stencil buffer
  void clearStencilGL();
  
  //! Enable or disable rendering to the color buffer
  void toggleColorRendering(bool b);
  //! Enable or disable writing into the depth buffer
  void toggleDepthRendering(bool b);
  void toggleDepthTest(bool b);
      
  //! Enables stencil tests, calls glStencilFunc
  void enableStencil(GLenum rel, GLint value, GLuint mask=0xffffffff);
  //! Specify what to do when the stencil and depth tests fail/succeed
  void setStencilOp(GLenum fail, GLenum zfail, GLenum zpass);
  void disableStencil();
  
  //! Specify alpha tests
  void setAlphaTest(GLenum function, float ref);

  inline int getScreenWidth() const { return width; }
  inline int getScreenHeight() const { return height; }
  
  void setClearColor(unsigned char, unsigned char, unsigned char);
  void setClearStencil(GLint);
  unsigned int getStencilBits() const;

  void setFog(double density, double start, double end);
  void disableFog();

private:
  enum State { Constructed, Initialized };
  State my_state;

  top10::util::Ref<CameraNode> camera;
  RenderingFeatures features;
  int width;
  int height;

  float clear_color[4];

  enum RenderMethod { Scene=0, Skybox, Hud };
  void renderCommon(RenderMethod, const RenderList&);

  //! Forbidden
  Renderer& operator=(const Renderer&);
  //! Forbidden
  Renderer(const Renderer&);
};

};
};

#endif
