/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/
#include "Renderer.hh"
#include "Node.hh"
#include "CameraNode.hh"
#include <GL/glu.h>

namespace top10 {
namespace graphX {

Renderer::Renderer(): my_state(Constructed), camera(0)
{
}

void Renderer::buildList(const Node* node, RenderList *rl) const
{
  assert(camera.getPtr());

  RenderState rs;
  node->buildRenderList(rs, rl, *camera);
}

void Renderer::initGL(int w, int h)
{
  assert(Initialized != my_state);

  my_state = Initialized;
  width = w;
  height = h;
  
  setClearColor(0, 0, 0);
  setClearStencil((1<<getStencilBits())/2);
  glEnable(GL_DEPTH_TEST);
  glDepthFunc(GL_LESS);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glDisable(GL_LIGHTING);
  glColor4f(1.0, 1.0, 1.0, 1.0);
  glViewport(0, 0, width, height);
}

void Renderer::resizeGL(int w, int h)
{
  width = w;
  height = h;
  glViewport(0, 0, width, height);
}

void Renderer::clearAllGL()
{
  glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT|GL_STENCIL_BUFFER_BIT);  
}

void Renderer::clearDepthGL()
{
  glClear(GL_DEPTH_BUFFER_BIT);  
}

void Renderer::clearColorGL()
{
  glClear(GL_COLOR_BUFFER_BIT);
}

void Renderer::clearStencilGL()
{
  glClear(GL_STENCIL_BUFFER_BIT);  
}

void Renderer::renderCommon(RenderMethod method, const RenderList& rl)
{
  assert(Initialized == my_state);
  assert(glGetError() == GL_NO_ERROR);

  // Set the initial state
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  if (method != Hud) {
    assert(camera.getPtr());
    gluPerspective(camera->getView().getFOV(), camera->getView().getRatio(), camera->getView().getNear(), camera->getView().getFar());
  }
  else {
    gluOrtho2D(-1.0, 1.0, -1.0, 1.0);   
  }
  
  glMatrixMode(GL_MODELVIEW);
  double M[16];
  switch (method) {
  case Skybox: camera->getView().getOrientGLMatrix(M); break;
  case Scene:  camera->getView().getGLMatrix(M); break;
  case Hud:    top10::math::Identity4().toGL(M); break;
  }
  glLoadMatrixd(M);
  glPushMatrix();  

  // Set the initial state
  RenderState initial, prev_state;
  initial.setStateGL(initial, features, true /* force_init */);
  prev_state = initial;
    
  for (RenderList::const_iterator it = rl.begin(); it != rl.end(); ++it) {
    // Set the state
    it->first.setStateGL(prev_state, features);
    
    // Render
    it->second->renderGL(features, it->first, *camera);
    
    glFlush();

    // Update the previous state
    prev_state = it->first;
    
#ifndef NDEBUG
    if (!prev_state.isExternal())
      prev_state.checkConsistency();
#endif
  }
  
  initial.setStateGL(initial, features, true);
  
  glMatrixMode(GL_MODELVIEW);
  glPopMatrix();
}
    
void Renderer::setCamera(CameraNode*c)
{
  camera = c;
  camera->setRatio((double)width/height);
}

CameraNode* Renderer::getCamera() const
{
  return camera.getPtr();
}

void Renderer::enableStencil(GLenum rel, GLint value, GLuint mask)
{
  glEnable(GL_STENCIL_TEST);
  glStencilFunc(rel, value, mask);    
}

void Renderer::setStencilOp(GLenum fail, GLenum zfail, GLenum zpass)
{
  glStencilOp(fail, zfail, zpass); 
}

void Renderer::disableStencil()
{
  glDisable(GL_STENCIL_TEST); 
}

void Renderer::toggleColorRendering(bool b)
{
  if (b) glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
  else glColorMask(GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE); 
}

void Renderer::toggleDepthRendering(bool b)
{
  if (b) glDepthMask(GL_TRUE);  
  else glDepthMask(GL_FALSE);
}

void Renderer::toggleDepthTest(bool b)
{
  if (b) glEnable(GL_DEPTH_TEST);
  else glDisable(GL_DEPTH_TEST);  
}

void Renderer::setAlphaTest(GLenum function, float ref)
{
  glEnable(GL_ALPHA_TEST);
  glAlphaFunc(function, ref);
}

void Renderer::setClearColor(unsigned char r, unsigned char g, unsigned char b)
{
  clear_color[0] = r/255.0f;
  clear_color[1] = g/255.0f;
  clear_color[2] = b/255.0f;
  clear_color[3] = 1.0f;

  glClearColor(clear_color[0], clear_color[1], clear_color[2], clear_color[2]);
  glFogfv(GL_FOG_COLOR, clear_color);
}

void Renderer::setClearStencil(GLint v)
{
  glClearStencil(v);
}

unsigned int Renderer::getStencilBits() const
{
  GLint ret = 0;
  glGetIntegerv(GL_STENCIL_BITS, &ret);
  return (unsigned int)ret;
}

void Renderer::setFog(double density, double start, double end)
{
  glFogi(GL_FOG_MODE, GL_LINEAR);		// Fog Mode
  glFogfv(GL_FOG_COLOR, clear_color);			// Set Fog Color
  glFogf(GL_FOG_DENSITY, density);				// How Dense Will The Fog Be
  glHint(GL_FOG_HINT, GL_DONT_CARE);			// Fog Hint Value
  glFogf(GL_FOG_START, start);				// Fog Start Depth
  glFogf(GL_FOG_END, end);				// Fog End Depth
  glEnable(GL_FOG);
}
 
void Renderer::disableFog()
{
  glDisable(GL_FOG);
}

};
};
