/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_GRAPHX_RENDERSTATE_HH
#define TOP10_GRAPHX_RENDERSTATE_HH

#include "math/Vertex.hh"
#include "math/Matrix.hh"
#include "Light.hh"
#include "RenderingFeatures.hh"

#include <vector>

#undef None

namespace top10 {
  namespace graphX {

    /*! RenderState is used to keep track of the current rendering state to avoid costly state changes. */
    class RenderState {
    public:
      typedef std::vector<Light> Lights;
      enum Filtering { None=0, Linear, Bilinear, Trilinear, Anisotropic };
      enum Culling { NoCulling=0, Front=1, Back=2 };
      enum PolyFillMode { Full=0, Lines=1 };
      enum FixedBits { MaterialBit=1, TextureBit=2, CullingBit=4, LineBit=8, PolyBit=16, LightsBit=32 };
      typedef signed long FixedBitsField;
      enum TexCoordGenMode { Explicit=0, ObjectLinear=1, EyeLinear=2, Spherical=3, Normal=4, Reflective=5 };
      enum TexEnvMode { Replace, Modulate, Decal, Blend };

    private:
      struct TextureUnitState
      {
	TextureUnitState();

	enum Type { Disabled, Dimension1, Dimension2, Dimension3, CubeMap };

	unsigned int id;
	Type type;
	TexEnvMode env_mode;
	//! Texture coordinates generation mode
	TexCoordGenMode gen_mode;
	//! Texture filtering
	Filtering filtering;
	//! Texture transform
	top10::math::Matrix4 transform;
      };

    public:
      RenderState();

      //! Check consistency between this render state and the actual OpenGL state
      void checkConsistency() const;
      
      inline void setTransform(const top10::math::Matrix4& M) { to_world = M; }
      inline const top10::math::Matrix4& getTransform() const { return to_world; }

      inline void setTextureTransform(const top10::math::Matrix4& M) {
	if ((fixed_flags & TextureBit) == 0)
	  texture_units.at(texture_unit).transform = M;
      }
      inline const top10::math::Matrix4& getTextureTransform() const {
	return texture_units.at(texture_unit).transform;
      }

      void addLight(Light);
      inline const Lights* getLights() const { return &lights; }
      inline void disableLights() { if ((fixed_flags & LightsBit) == 0) lights.clear(); }
      
      // Accessors
      inline void setAlpha(unsigned char c) {
        if ((fixed_flags & MaterialBit) == 0) alpha = c;
      }
      inline void setColor(unsigned char _r, unsigned char _g, unsigned char _b) {
        if ((fixed_flags & MaterialBit) == 0) { r=_r; g=_g; b=_b; }
      }
      inline void setGlow(unsigned char _r, unsigned char _g, unsigned char _b) {
        if ((fixed_flags & MaterialBit) == 0) { glow_r = _r; glow_b = _b; glow_g = _g; }
      }
      inline void setTextureId(unsigned int id) {
        if ((fixed_flags & TextureBit) == 0)
	{
	  texture_units.at(texture_unit).id = id;
	  texture_units.at(texture_unit).type = TextureUnitState::Dimension2;
	}
      }
      inline void setCubeMapId(unsigned int id)
      {
	if ((fixed_flags & TextureBit) == 0)
	{
	  texture_units.at(texture_unit).id = id;
	  texture_units.at(texture_unit).type = TextureUnitState::CubeMap;
	}
      }
      inline void setTexEnv(TexEnvMode mode)
      {
	if ((fixed_flags & TextureBit) == 0)
	{
	  texture_units.at(texture_unit).env_mode = mode;
	}
      }
      inline void setFiltering(Filtering f) {
        if ((fixed_flags & TextureBit) == 0)
	{
	  texture_units.at(texture_unit).filtering = f;
	}
      }
      inline void setOffset(float f) {
        offset = f;
      }
      inline void setCulling(Culling c) {
        if ((fixed_flags & CullingBit) == 0) { culling = c; }
      }
      inline void setFront(bool b) {
        if ((fixed_flags & CullingBit) == 0) { front_is_direct = b; }
      }
      inline void setLineWidth(float f) {
        if ((fixed_flags & LineBit) == 0) { line_width = f; }
      }
      inline void setPolyFill(PolyFillMode m) {
        if ((fixed_flags & PolyBit) == 0) { poly_fill_mode = m; }
      }
      inline void setTexGenMode(TexCoordGenMode m) {
	if ((fixed_flags & TextureBit) == 0)
	{
	  texture_units.at(texture_unit).gen_mode = m;
	}
      }
      inline void setTextureUnit(int u)
      {
	texture_unit = u;
      }
      inline int getTextureUnit()
      {
	return texture_unit;
      }

      inline Culling getCulling() const { return culling; }
      
      inline void disableChanges(FixedBitsField bits) { fixed_flags |= bits; }
      inline void disableChanges() { fixed_flags = (signed long)-1; }
      
      inline void enableChanges(FixedBitsField bits) { fixed_flags &= ~bits; }
      inline void enableChanges() { fixed_flags = 0; }
      
      inline void setChanges(FixedBitsField f) { fixed_flags = f; }
   
      inline void toggleExternal(bool b)
      {
	if (b)
	{
	  is_external = true;
	  disableChanges();
	}
	else
	{
	  is_external = false;
	  enableChanges();
	}
      }

      inline bool isExternal() const
      {
	return is_external;
      }

      //! Comparison operator, useful to sort RenderList according to the state
      bool operator<(const RenderState&) const;

      //! Update the current OpenGL state
      /*! \param old_state old OpenGL state, used to make as few state changes as possible. */
      void setStateGL(const RenderState& old_state, const RenderingFeatures&, bool force_init = false) const;
      
    private:
      //! The size of texture_units.
      static const unsigned int MAX_TEXTURE_UNITS = 32;

      //! If true, the object to be rendered uses its own state management.
      bool is_external;

      //! The texture unit on which nodes affecting the texture state will work.
      /*! That attribute is not part of the OpenGL state. */
      int texture_unit;

      //! Specifies which attributes are not allowed to be set
      FixedBitsField fixed_flags;
      //! Opacity
      unsigned char alpha;
      //! ambient and diffuse color
      unsigned char r,g,b;
      //! Emission (glow)
      unsigned char glow_r, glow_g, glow_b;
      //! The state of each texture unit.
      std::vector<TextureUnitState> texture_units;
      //! Depth offset
      float offset;
      //! Which faces should be drawn
      Culling culling;
      //! What is the front face
      bool front_is_direct;
      //! Line width
      float line_width;
      //! If polygons should be filled, or drawn in wireframes
      PolyFillMode poly_fill_mode;
      //! ModelView transform
      top10::math::Matrix4 to_world;
      //! Lights
      Lights lights;
    };
  };
};

#endif
