/*
  Top10, a racing simulator
  Copyright (C) 2000-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#ifdef WIN32
#include <windows.h>
#endif

#include "RenderState.hh"
#include "TextureManager.hh"
#include "extras/GLee.h"
#include <cassert>

using namespace top10::graphX;

RenderState::TextureUnitState::TextureUnitState()
: id(0), type(Disabled), env_mode(Modulate), gen_mode(Explicit), filtering(None)
{
  transform = top10::math::Identity4();
}

RenderState::RenderState():
  is_external(false),
  texture_unit(0),
  fixed_flags(0),
  alpha(255), r(0), g(0), b(0), glow_r(0), glow_g(0), glow_b(0), offset(0.0),
  culling(NoCulling), front_is_direct(true), line_width(1.0), poly_fill_mode(Full)
{
  to_world = top10::math::Identity4();

  TextureUnitState tex_unit;
  texture_units.reserve(MAX_TEXTURE_UNITS);
  texture_units.insert(texture_units.end(), MAX_TEXTURE_UNITS, tex_unit);
}

void RenderState::addLight(Light l)
{
  if ((fixed_flags & LightsBit) == 0)
    lights.push_back(l);  
}

bool RenderState::operator<(const RenderState& other) const
{
  // Sort according to is_external
  if (is_external != other.is_external)
    return !is_external;

  // 0 Sort according to transparency (draw opaque objects first)
  // FIXME: drawing opaque object first should not be handled here.
  if (alpha != other.alpha) return other.alpha < alpha;
  
  // 1 Sort wrt textures
  {
    std::vector<TextureUnitState>::const_iterator it2 = other.texture_units.begin();
    for (std::vector<TextureUnitState>::const_iterator it = texture_units.begin();
      it != texture_units.end();
      ++it, ++it2)
    {
      if ((it->type == TextureUnitState::Disabled) != (it2->type == TextureUnitState::Disabled))
	return it->type == TextureUnitState::Disabled;

      if (it->id != it2->id)
	return it->id < it2->id;
      
      if (it->env_mode != it2->env_mode)
	return it->env_mode < it2->env_mode;

      if (it->gen_mode != it2->gen_mode)
	return it->gen_mode < it2->gen_mode;

      // Texture transform
      for (int j=0; j<4; ++j)
      {
	// We assume the last line is always (0 0 0 1)
	for (int i=0; i<3; ++i) {
	  if (it->transform(i,j) != it2->transform(i,j))
	    return it->transform(i,j) < it->transform(i,j);
	}
      }

      if (it->filtering != it2->filtering)
	return it->filtering < it2->filtering;

      if (it->type != it2->type)
	return it->type < it2->type;
    }
  }

  // 2 Sort wrt material props
#define TMP_SORT(r)\
  if (r != other.r) return r < other.r
  TMP_SORT(r);
  TMP_SORT(g);
  TMP_SORT(b);
  TMP_SORT(glow_r);
  TMP_SORT(glow_g);
  TMP_SORT(glow_b);
#undef TMP_SORT

  // 3. lighting environment.
  Lights::const_iterator it1, it2;
  it1 = lights.begin();
  it2 = other.lights.begin();
  while (it1 != lights.end() && it2 != other.lights.end()) {
    if (*it1 != *it2) return *it1 < *it2;
    ++it1;
    ++it2;
  }
  if (it1 != lights.end()) return false;
  if (it2 != other.lights.end()) return true;
  
  // 4 transforms.
  for (int j=0; j<4; ++j) {
    // We assume the last line is always (0 0 0 1)
    for (int i=0; i<3; ++i) {
      if (to_world(i,j) != other.to_world(i,j)) return to_world(i,j) < other.to_world(i,j);
    }
  }
    
  // 5 line width
  if (other.line_width != line_width) return other.line_width < line_width;
  
  // 6 fill mode
  if (other.poly_fill_mode != poly_fill_mode) return other.poly_fill_mode < poly_fill_mode;
  
  // 7 depth offset  
  if (other.offset != offset)
    return other.offset < offset; // It makes sense to draw triangles with big offsets first.
  
  // 8 culling
  if (other.front_is_direct != front_is_direct) return front_is_direct;
  if (other.culling != culling)
    return culling < other.culling;

  return false;
}

void RenderState::setStateGL(const RenderState& old_state, const RenderingFeatures& features, bool force_init) const
{
  assert(glGetError() == GL_NO_ERROR);

#if 0
  // Reset to the initial state if the state is now going to be affected by direct interaction with OpenGL.
  if (is_external)
  {
    RenderState initial;
    initial.setStateGL(old_state, features, true);
    if (features.hasMultiTexturing)
      glActiveTextureARB(GL_TEXTURE0_ARB);
    return;
  }
#endif

  // Was the old_state external?
  if (old_state.is_external)
    force_init = true; // Reset all settings, we can't trust the OpenGL state matches old_state

  // Texture mapping
  {
    unsigned int i_unit = 0;
    for (std::vector<TextureUnitState>::const_iterator it = texture_units.begin(), it2 = old_state.texture_units.begin();
      it != texture_units.end() && (!features.hasMultiTexturing || i_unit < features.maxMultiTextures);
      ++it, ++it2, ++i_unit)
    {
      if (features.hasMultiTexturing)
	glActiveTextureARB(GL_TEXTURE0_ARB + i_unit);

      assert(glGetError() == GL_NO_ERROR);

      TextureUnitState::Type type = it->type;
      if (type == TextureUnitState::CubeMap && !features.hasCubeMaps)
	type = TextureUnitState::Disabled;

      TextureUnitState::Type type2 = it2->type;
      if (type2 == TextureUnitState::CubeMap && !features.hasCubeMaps)
	type2 = TextureUnitState::Disabled;

      bool do_init = force_init || (type != TextureUnitState::Disabled && type2 == TextureUnitState::Disabled)
	|| (type != TextureUnitState::Disabled && it->id != it2->id);

      GLenum target;
      switch (type)
      {
	case TextureUnitState::Disabled:   break;
	case TextureUnitState::Dimension1: target = GL_TEXTURE_1D; break;
	case TextureUnitState::Dimension2: target = GL_TEXTURE_2D; break;
	case TextureUnitState::Dimension3: target = GL_TEXTURE_3D; break;
	case TextureUnitState::CubeMap:    if (features.hasCubeMaps) target = GL_TEXTURE_CUBE_MAP_EXT; break;
      }

      if (do_init || type != type2)
      {
	// Disable higher priority modes
	// Each case goes through all following cases.
	switch (type)
	{
	case TextureUnitState::Disabled:   glDisable(GL_TEXTURE_1D); glBindTexture(GL_TEXTURE_1D, 0);
	case TextureUnitState::Dimension1: glDisable(GL_TEXTURE_2D); glBindTexture(GL_TEXTURE_2D, 0);
	case TextureUnitState::Dimension2: glDisable(GL_TEXTURE_3D); glBindTexture(GL_TEXTURE_3D, 0);
	case TextureUnitState::Dimension3: if (features.hasCubeMaps)
					   {
					     glDisable(GL_TEXTURE_CUBE_MAP_EXT);
					     glBindTexture(GL_TEXTURE_CUBE_MAP_EXT, 0);
					   }
	case TextureUnitState::CubeMap:    break; /* Nothing to do */
	}
	assert(glGetError() == GL_NO_ERROR);

	// Enable the mode we want
	if (type != TextureUnitState::Disabled)
	{
	  glEnable(target);
	  assert(glGetError() == GL_NO_ERROR);
	}
      }

      // The id of the texture object
      if ((do_init || it->id != it2->id) && type != TextureUnitState::Disabled)
      {
	glBindTexture(target, it->id);
	assert(glGetError() == GL_NO_ERROR);
      }

      if ((do_init || it->env_mode != it2->env_mode) && type != TextureUnitState::Disabled)
      {

	switch (it->env_mode)
	{
	case Replace:  glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE); break; 
	case Modulate: glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE); break; 
	case Decal:    glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL); break; 
	case Blend:    glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_BLEND); break; 
	}
	assert(glGetError() == GL_NO_ERROR);
      }

      // Filtering mode
      if ((do_init || it->filtering != it2->filtering) && type != TextureUnitState::Disabled)
      {
	switch(it->filtering)
	{
	case None:
	  glTexParameteri(target, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	  glTexParameteri(target, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	  break;
	case Linear:
	  glTexParameteri(target, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	  glTexParameteri(target, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	  break;
	case Bilinear:
	  glTexParameteri(target, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	  glTexParameteri(target, GL_TEXTURE_MIN_FILTER, GL_NEAREST_MIPMAP_LINEAR);
	  break;
	case Trilinear:
	  glTexParameteri(target, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	  glTexParameteri(target, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
	  break;
	case Anisotropic:
	  if (!features.hasAnisotropicFiltering) {
	    glTexParameteri(target, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	    glTexParameteri(target, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	  }
#ifdef GL_TEXTURE_MAX_ANISOTROPY_EXT
	  else {      
	    glTexParameterf(target, GL_TEXTURE_MAX_ANISOTROPY_EXT, features.maxAnisotropy);
	  }
#else
#warning "GL_TEXTURE_MAX_ANISOTROPY_EXT undefined"
#endif
	  break;
	}
	assert(glGetError() == GL_NO_ERROR);
      }

      // Texture transform
      if (do_init || it->transform != it2->transform)
      {
	glMatrixMode(GL_TEXTURE);
	double d[16];
	it->transform.toGL(d);
	glLoadMatrixd(d);
	assert(glGetError() == GL_NO_ERROR);
      }

      // Texture coordinate generation mode
      if (do_init || it->gen_mode != it2->gen_mode)
      {
	if (it->gen_mode == Explicit)
	{
	  glDisable(GL_TEXTURE_GEN_S);
	  glDisable(GL_TEXTURE_GEN_T);
	  glDisable(GL_TEXTURE_GEN_R);
	}
	else
	{
	  glEnable(GL_TEXTURE_GEN_S);
	  glEnable(GL_TEXTURE_GEN_T);
	  glEnable(GL_TEXTURE_GEN_R);
	}

	switch (it->gen_mode)
	{
	case Explicit:
	  // Nothing to do
	  break;

	case ObjectLinear:
	  glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);
	  glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);
	  glTexGeni(GL_R, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);
	  break;

	case EyeLinear:
	  glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_EYE_LINEAR);
	  glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, GL_EYE_LINEAR);
	  glTexGeni(GL_R, GL_TEXTURE_GEN_MODE, GL_EYE_LINEAR);
	  break;

	case Spherical:
	  glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_SPHERE_MAP);
	  glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, GL_SPHERE_MAP);
	  glTexGeni(GL_R, GL_TEXTURE_GEN_MODE, GL_SPHERE_MAP);
	  break;

	case Normal:
	  glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_NORMAL_MAP_EXT);
	  glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, GL_NORMAL_MAP_EXT);
	  glTexGeni(GL_R, GL_TEXTURE_GEN_MODE, GL_NORMAL_MAP_EXT);
	  break;

	case Reflective:
	  glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_REFLECTION_MAP_EXT);
	  glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, GL_REFLECTION_MAP_EXT);
	  glTexGeni(GL_R, GL_TEXTURE_GEN_MODE, GL_REFLECTION_MAP_EXT);
	  break;
	}
	assert(glGetError() == GL_NO_ERROR);
      }

    }
  }
  
  if (is_external && features.hasMultiTexturing)
    glActiveTextureARB(GL_TEXTURE0_ARB);

  // Material
  if (force_init || r != old_state.r || g != old_state.g || b != old_state.b || alpha != old_state.alpha) {
    float v[] = {r/255.0, g/255.0, b/255.0, alpha/255.0};
#ifndef NDEBUG
    for (int i=0; i<4; ++i) {
      assert(v[i] >= 0.0);
      assert(v[i] <= 1.0);
    }
#endif     
    glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, v);
    glColor4f(v[0], v[1], v[2], v[3]);
    assert(glGetError() == GL_NO_ERROR);
  }

  if (force_init || glow_r != old_state.glow_r || glow_g != old_state.glow_g || glow_b != old_state.glow_b || alpha != old_state.alpha) {
    float v[] = {glow_r/255.0, glow_g/255.0, glow_b/255.0, alpha/255.0};
#ifndef NDEBUG
    for (int i=0; i<4; ++i) {
      assert(v[i] >= 0.0);
      assert(v[i] <= 1.0);
    }
#endif         
    glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, v);
    assert(glGetError() == GL_NO_ERROR);
  }
    
  // Lighting
  if ((force_init || !old_state.lights.empty()) && lights.empty())
    glDisable(GL_LIGHTING);
  else if ((force_init || old_state.lights.empty()) && !lights.empty())
    glEnable(GL_LIGHTING);

  //  Lights use global coordinates
  glMatrixMode(GL_MODELVIEW);
  glPopMatrix();
  // Transform is now set to the inverse of the camera's transform

  Lights::const_iterator it1 = lights.begin();
  Lights::const_iterator it2 = old_state.lights.begin();
  int i=0;
  while (it1 != lights.end() && it2 != old_state.lights.end()) {
    it1->setLightGL(GL_LIGHT0 +i);
    ++it1;
    ++it2;
    ++i;
  }

  while (it1 != lights.end()) {
    it1->setLightGL(GL_LIGHT0 +i);
    ++it1;
    ++i;
  }
  while (it2 != old_state.lights.end()) {
    glDisable(GL_LIGHT0 +i);
    ++it2;
    ++i;  
  }
  assert(glGetError() == GL_NO_ERROR);
  
  {
    glPushMatrix(); /* <- That copies the matrix that was at the top,
                        which is the transformation generated by the camera */
    double d[16];
    to_world.toGL(d);
    glMultMatrixd(d);
    assert(glGetError() == GL_NO_ERROR);
  }

  // Line width
  if (force_init || old_state.line_width != line_width) {
    glLineWidth(line_width);
    assert(glGetError() == GL_NO_ERROR);
  }
  
  // Fill mode
  if (force_init || old_state.poly_fill_mode != poly_fill_mode) {
    glPolygonMode(GL_FRONT_AND_BACK, poly_fill_mode==Full? GL_FILL:GL_LINE);
    assert(glGetError() == GL_NO_ERROR);
  }
  
  // Depth Offset
  if (force_init || old_state.offset != offset) {
    if (offset != 0.0) {
      glEnable(GL_POLYGON_OFFSET_FILL);
      glPolygonOffset(0, offset);
    }
    else {
      glDisable(GL_POLYGON_OFFSET_FILL);
    }
    assert(glGetError() == GL_NO_ERROR);
  }
  
  // Culling
  if (force_init || old_state.front_is_direct != front_is_direct) {
    if (front_is_direct) glFrontFace(GL_CCW);
    else glFrontFace(GL_CW);
    assert(glGetError() == GL_NO_ERROR);
  }
  
  if (force_init || old_state.culling != culling) {
    if (culling == NoCulling)
      glDisable(GL_CULL_FACE);
    else {
      glEnable(GL_CULL_FACE);
      if (culling == Front) glCullFace(GL_FRONT);
      else glCullFace(GL_BACK);
    }
    assert(glGetError() == GL_NO_ERROR);
  }
  
#ifndef NDEBUG
  checkConsistency();
#endif

  glFlush();
}

void RenderState::checkConsistency() const
{
  float v[4];
  glGetMaterialfv(GL_FRONT, GL_AMBIENT, v);
  float _r = v[0];
  float _g = v[1];
  float _b = v[2];
  float _alpha = v[3];
  
  glGetMaterialfv(GL_FRONT, GL_EMISSION, v);
  float _glow_r = v[0];
  float _glow_g = v[1];
  float _glow_b = v[2];
  float _alpha2 = v[3];
  
  std::vector<float> light_r, light_g, light_b, light_a;
  for (unsigned int i=0; i<lights.size(); ++i) {
    GLboolean light_enabled;
    glGetBooleanv(GL_LIGHT0+i, &light_enabled);
    if (!light_enabled) {
      std::cerr<<"Light "<<i<<" disabled, but should be enabled."<<std::endl;
      abort();
    }
    glGetLightfv(GL_LIGHT0+i, GL_DIFFUSE, v);
    light_r.push_back(v[0]);
    light_g.push_back(v[1]);
    light_b.push_back(v[2]);
    light_a.push_back(v[3]);
  }
  GLboolean lighting;
  glGetBooleanv(GL_LIGHTING, &lighting);
  if (lights.empty() != !lighting)
  {
    std::cerr<<"Lighting "<<lighting<<" lights.empty "<<lights.empty()<<std::endl;
    abort();
  }

  bool passed = true;  

#define CHECK_CONSIST(name, value, expected)\
if ((value) !=  (expected)) {\
  std::cerr<<"Consistency check for "<<name<<" failed. Expected "<<expected<<", got "<<value<<std::endl;\
  passed = false;\
}

#define ROUND(f) (((int)((f)*100.0))/100.0)

  CHECK_CONSIST("red", ROUND(_r), ROUND(r/255.0));
  CHECK_CONSIST("green", ROUND(_g), ROUND(g/255.0));
  CHECK_CONSIST("blue", ROUND(_b), ROUND(b/255.0));
  CHECK_CONSIST("glow red", ROUND(_glow_r), ROUND(glow_r/255.0));
  CHECK_CONSIST("glow green", ROUND(_glow_g), ROUND(glow_g/255.0));
  CHECK_CONSIST("glow blue", ROUND(_glow_b), ROUND(glow_b/255.0));
  CHECK_CONSIST("alpha", ROUND(_alpha), ROUND(alpha/255.0));  
  
  for (unsigned int i=0; i<lights.size(); ++i) {
    CHECK_CONSIST("Light red", ROUND(light_r.at(i)), 1.0);
    CHECK_CONSIST("Light green", ROUND(light_g.at(i)), 1.0);
    CHECK_CONSIST("Light blue", ROUND(light_b.at(i)), 1.0);
    CHECK_CONSIST("Light alpha", ROUND(light_a.at(i)), 1.0);
  }
  
#undef ROUND
#undef CHECK_CONSIST
  if (!passed) {
    abort();
  }
}
