/*
  Top 10, a racing simulator
  Copyright (C) 2003,2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_GRAPHXPROXYNODE_HH
#define TOP10_GRAPHXPROXYNODE_HH

#include "MeshNode.hh"
#include "TransformNode.hh"

namespace top10 {
namespace graphX {

class ProxyOperation: public top10::util::RefCount
{
public:
  virtual Node* makeChild(const MeshNode*) const =0;
  
  virtual ~ProxyOperation() {}
};

//! Virtually duplicate a node hierarchy
/*! Each transform node is duplicated (using TransformNodeProxy),
  each MeshNode is replaced according to the provided ProxyOperation,
  all other node types are duplicated by dummies that don't change the render state, nor render anything.
  */
class NodeProxy: public top10::graphX::Node {  
public:
  /*!
  \param store if true, store and reuse duplicata, thus saving time. Not suitable if the original hierarchy changes.
  */
  NodeProxy(const top10::graphX::Node*, const ProxyOperation*, bool store = false);
    
  top10::graphX::Node* getChild(int idx) const;
  
  //! If you want to add children, don't do it through the proxy.
  inline void addChild(top10::graphX::Node*) {}
  //! Don't remove children through the proxy.
  inline void removeChild(top10::graphX::Node*) {}
  //! Do nothing
  void renderGL(const RenderingFeatures&, const RenderState&, const CameraNode&) const {}
  
  inline const Node* getOriginal() const { return original.getPtr(); }
  
private:
  top10::util::Ref<const Node> original;
  top10::util::Ref<const ProxyOperation> op;
  bool store;
  mutable NodeRefs stored;
  
  Node* makeChild(int idx) const;
};

class TransformNodeProxy: public NodeProxy {
public:
  TransformNodeProxy(const top10::graphX::TransformNode*, const ProxyOperation*, bool store = false);
  
  inline top10::math::Matrix4 toWorld() const { return original->toWorld(); }
  
protected:
  void modifyRenderState(top10::graphX::RenderState& s, const CameraNode&) const;
  
private:
  top10::util::Ref<const top10::graphX::TransformNode> original;
};

inline Node* makeProxy(Node* n, const ProxyOperation* op, bool store = false)
{
  TransformNode* t = dynamic_cast<TransformNode*>(n);
  MeshNode* m = dynamic_cast<MeshNode*>(n);
  
  if (t)
    return new TransformNodeProxy(t, op, store);
  else if (m)
    return op->makeChild(m);
  else
    return new NodeProxy(n, op, store);
}

}
}

#endif
