/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#ifndef TOP10_GRAPHX_NODE_HH
#define TOP10_GRAPHX_NODE_HH

#include "math/Matrix.hh"
#include "util/RefCount.hh"
#include "RenderState.hh"
#include "RenderList.hh"
#include "RenderingFeatures.hh"

#include <string>

#define for_all_children(n, child) do { int i=0; for (top10::graphX::Node* child = (n)->getChild(0); child; child = (n)->getChild(++i))
#define for_all_children_end } while (0)

namespace top10 {
  namespace graphX {

    class CameraNode;

    class Node: public top10::util::RefCount {
    public:
      Node();

      //! Return a pointer to the idx-th child, or NULL if idx is out of bound
      virtual Node* getChild(int idx) const =0;

      //! Add a child to this node
      virtual void addChild(Node*) =0;

      //! Remove a child.
      virtual void removeChild(Node*) =0;
      
      virtual Node* find(std::string);
      
      //! Build the multimap from RenderStates to nodes
      /*! By default: calls modifyRenderState, then call buildRenderList on all children */
      virtual void buildRenderList(RenderState, RenderList*, const CameraNode&) const;

      //! Execute OpenGL instructions to render this node.
      virtual void renderGL(const RenderingFeatures&, const RenderState&, const CameraNode&) const =0;

      inline std::string getName() const { return name; }
      inline void setName(std::string s) { name = s; }

      inline void disableChanges(RenderState::FixedBitsField bits) { fixed_flags |= bits; }
      inline void disableChanges() { fixed_flags = (signed long)-1; }

      virtual ~Node() {};

    protected:
      virtual void modifyRenderState(RenderState&, const CameraNode&) const {};
      
    private:
      std::string name;
      RenderState::FixedBitsField fixed_flags;
    };

    typedef top10::util::Ref< Node > NodeRef;
    typedef std::vector< NodeRef > NodeRefs;
  };
};

#endif
