/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#ifndef TOP10_GRAPHX_MESHNODE_HH
#define TOP10_GRAPHX_MESHNODE_HH

#include "math/Mesh.hh"
#include "math/Box.hh"
#include "LeafNode.hh"
#include "util/NoCopy.hh"

#include <vector>

namespace top10 {
  namespace graphX {

    struct MeshNodePrivate;

    //! Extends math::Mesh with texture mapping and normals
    class MeshNode: private top10::util::NoCopy, public LeafNode {
    public:
      //! Coordinates in a 2d texture
      struct TextureCoord {
	TextureCoord(double _s, double _t): s(_s), t(_t) {}
	TextureCoord(): s(0.0), t(0.0) {}

	double s, t;
      };

    public:
      /*!
	\param mesh pointer to the triangles
	\param normals array of normals
	\param normal_idxs indices into the normal array. Three for each triangle.
	\param tex_coords array of texture coordinates. 3 for each triangle.
      */
      MeshNode(const top10::math::Mesh* mesh,
	       std::vector<top10::math::Vector> normals,
	       std::vector<int> normal_idxs,
	       std::vector<TextureCoord> tex_coords,
               std::string name);
      /*!
        \param normals_size must be mesh->getVertices()->size() *3
        \param normals array of normals of size normals_size
        \param tex_coords_size must be mesh->getVertices()->size() *2
        \param tex_coords array of texture coordinates of size tex_coords_size
      */
      MeshNode(const top10::math::Mesh* mesh,
               int normals_size,
               const float* normals,
               int tex_coords_size,
               const float* tex_coords,
               std::string name);

      //! Compute smooth normals automatically
      MeshNode(const top10::math::Mesh* mesh,
	       std::vector<TextureCoord> tex_coords,
               std::string name);

      ~MeshNode();

//      void buildRenderList(RenderState s, RenderList* rl, const CameraNode&) const;
      void renderGL(const RenderingFeatures&, const RenderState&, const CameraNode&) const;

      inline const std::vector<top10::math::Vector>* getNormals() const { return &normals; }
      inline const std::vector<TextureCoord>* getTexCoords() const { return &tex_coords; }
      inline const std::vector<int>* getNormalIndices() const { return &normal_idxs; }
      inline const top10::math::Mesh* getMesh() const { return mesh; }
      inline void setMaterialName(std::string n) { mat_name = n; }
      inline std::string getMaterialName() const { return mat_name; }
      inline void setShadowVolumeFlag(bool b) { shadow_volume_enabled = b; }
      inline bool getShadowVolumeFlag() const { return shadow_volume_enabled; }

    private:
      top10::math::Box getBoundingBox(const top10::math::Matrix4&) const;

    private:
      std::string mat_name;
      const top10::math::Mesh* mesh;
      std::vector<top10::math::Vector> normals;
      //! Three for each triangle
      std::vector<TextureCoord> tex_coords;
      //! Three for each triangle
      std::vector<int> normal_idxs;
      //! If true, shadow volumes are produced for this mesh.
      bool shadow_volume_enabled;

      MeshNodePrivate* data;
    };

    typedef top10::util::Ref< MeshNode > MeshNodeRef;
    typedef std::vector< MeshNodeRef > MeshNodeRefs;
  };
};

#endif
