/*
  Top10, a racing simulator
  Copyright (C) 2000-2005  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "extras/GLee.h"
#include "MeshNode.hh"
#include "CameraNode.hh"

using namespace top10::graphX;

struct top10::graphX::MeshNodePrivate
{
  GLuint disp_list;
};

MeshNode::MeshNode(const top10::math::Mesh* _mesh,
		   std::vector<top10::math::Vector> _normals,
		   std::vector<int> _normal_idxs,
		   std::vector<TextureCoord> _tex_coords,
                   std::string name):
  mesh(_mesh),
  normals(_normals),
  tex_coords(_tex_coords),
  normal_idxs(_normal_idxs),
  shadow_volume_enabled(true),
  data(0)
{
  setName(name);
}

MeshNode::MeshNode(const top10::math::Mesh* mesh,
                   int norms_size,
                   const float* norms,
                   int texs_size,
                   const float* texs,
                   std::string name):
  mesh(mesh),
  shadow_volume_enabled(true),
  data(0)
{
  // Build the vector of normals
  assert(norms_size == (int)mesh->getVertices()->size() *3);
  int n_normals = norms_size/3;
  
  normals.reserve(n_normals);
  normal_idxs.reserve((int)mesh->getFaces()->size()*3);
  
  for (int i=0; i<n_normals; ++i) {
    normals.push_back(top10::math::Vector(norms[i*3], norms[i*3+1], norms[i*3+2]));
  }
  for (int i=0; i<(int)mesh->getFaces()->size(); ++i) {
    normal_idxs.push_back(mesh->getFaces()->at(i).idxs[0]);
    normal_idxs.push_back(mesh->getFaces()->at(i).idxs[1]);
    normal_idxs.push_back(mesh->getFaces()->at(i).idxs[2]);
  }
  
  // Build a vector of texture coordinates
  assert(texs_size == (int)mesh->getVertices()->size() *2);
  tex_coords.reserve((int)mesh->getFaces()->size()*3);
  
  for (int i=0; i<(int)mesh->getFaces()->size(); ++i) {
    for (int j=0; j<3; ++j) {
      TextureCoord t(texs[2*mesh->getFaces()->at(i).idxs[j] +0],
	             texs[2*mesh->getFaces()->at(i).idxs[j] +1]);
      tex_coords.push_back(t);
    }
  }
  
  setName(name);
}
      
MeshNode::MeshNode(const top10::math::Mesh* _mesh,
		   std::vector<TextureCoord> _tex_coords,
                   std::string name):
  mesh(_mesh),
  tex_coords(_tex_coords),
  shadow_volume_enabled(true),
  data(0)
{
  using top10::math::Vector;
  using top10::math::Mesh;

  setName(name);
  
  const std::vector<Vector>* vertices = mesh->getVertices();
  const std::vector<Mesh::Face>* faces = mesh->getFaces();
  
  // One normal for each vertex
  normals.resize(vertices->size());
  // Three for each triangle
  normal_idxs.resize(faces->size() *3);

  // Sum the normal at each vertex over all faces sharing this vertex
  std::vector<int>::iterator idx_it = normal_idxs.begin();
  for (std::vector<Mesh::Face>::const_iterator face = faces->begin();
       face != faces->end();
       ++face) {
    int i0 = face->idxs[0], i1 = face->idxs[1], i2 = face->idxs[2];
    Vector N = (vertices->at(i1) - vertices->at(i0)) ^ (vertices->at(i2) - vertices->at(i1));
    double Ns = N.size();
    if (Ns > SMALL_VECTOR) {
      N/=Ns;
      normals.at(i0) += N;
      normals.at(i1) += N;
      normals.at(i2) += N;
    }
    *idx_it = i0; ++idx_it;
    *idx_it = i1; ++idx_it;
    *idx_it = i2; ++idx_it;
  }

  // Normalize to get the average
  for (std::vector<Vector>::iterator normal = normals.begin();
       normal != normals.end();
       ++normal) {
    double Ns = normal->size();
    if (Ns > SMALL_VECTOR)
      *normal /= Ns;
    else
      *normal = Vector(0,1,0);
  }
}
  
void MeshNode::renderGL(const RenderingFeatures& unused, const RenderState& rs, const CameraNode&) const
{
  using top10::math::Mesh;
  using top10::math::Vector;

  if (data)
  {
    glCallList(data->disp_list);
  }
  else
  {
    // Create the display list
    const_cast<MeshNode*>(this)->data = new MeshNodePrivate;
    const_cast<MeshNode*>(this)->data->disp_list = glGenLists(1);
    glNewList(data->disp_list, GL_COMPILE);

    const std::vector<Mesh::Face>* faces = mesh->getFaces();
    const std::vector<Vector>* vertices = mesh->getVertices();

    std::vector<int>::const_iterator normal_idx_it = normal_idxs.begin();
    std::vector<TextureCoord>::const_iterator texel_it = tex_coords.begin();
    glBegin(GL_TRIANGLES);
    for (std::vector<Mesh::Face>::const_iterator face_it = faces->begin();
      face_it != faces->end();
      ++face_it) {
	for (int i=0; i<3; ++i, ++normal_idx_it, ++texel_it) {
	  assert(normal_idx_it != normal_idxs.end());
	  assert(texel_it != tex_coords.end());

	  top10::math::Vector normal = normals.at(*normal_idx_it);
	  assert(normal.size2() > 0.9 && normal.size2() < 1.1);

	  glNormal3d(normal.x, normal.y, normal.z);

	  glTexCoord2d(texel_it->s, texel_it->t);

	  top10::math::Vector v = vertices->at(face_it->idxs[i]);
	  glVertex3d(v.x, v.y, v.z);
	}
    }
    glEnd();

    glEndList();
  }

#if 0
  // Debug: draw the bounding box
  top10::math::Vector v[8];
  top10::math::Identity4 I;
  getBoundingBox(I).getVertices(v);
#define DRAW(v) glVertex3f(v.x, v.y, v.z)
#define DRAW_LOOP(a,b,c,d)\
  glBegin(GL_LINE_LOOP);\
  DRAW(v[a]);\
  DRAW(v[b]);\
  DRAW(v[c]);\
  DRAW(v[d]);\
  glEnd();
  
  DRAW_LOOP(0,1,3,2);
  DRAW_LOOP(4,5,7,6);
  DRAW_LOOP(0,1,5,4);
  DRAW_LOOP(2,3,7,6);

#undef DRAW
#undef DRAW_LOOP
#endif
}

MeshNode::~MeshNode()
{
  if (data)
  {
    glDeleteLists(data->disp_list, 1);
  }

  delete data;
  data = 0;
}

top10::math::Box MeshNode::getBoundingBox(const top10::math::Matrix4& M) const
{
  using top10::math::Vector;

  const Vector center = 0.5*(mesh->getBoundMin() + mesh->getBoundMax());
  const Vector right = Vector(mesh->getBoundMax().x - center.x, 0.0, 0.0);
  const Vector up = Vector(0.0, mesh->getBoundMax().y - center.y, 0.0);
  const Vector deep = Vector(0.0, 0.0, mesh->getBoundMax().z - center.z);

  top10::math::Box bbox( center, right, up, deep );
  
  const Vector T = M.getColumn(3);
  top10::math::Matrix3 R(M.getColumn(0), M.getColumn(1), M.getColumn(2));
  bbox.rotate(R);
  bbox.translate(T);

  return bbox;
}
#if 0
void MeshNode::buildRenderList(RenderState s, RenderList* rl, const CameraNode& camera) const
{
  top10::math::Box bbox = getBoundingBox(s.getTransform());

  if (camera.getView().overlap(bbox) != top10::helpers::None)
    LeafNode::buildRenderList(s, rl, camera);
}
#endif