/*
  Top10, a racing simulator
  Copyright (C) 2000-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#ifndef TOP10_GRAPHX_EXTRUDER_HH
#define TOP10_GRAPHX_EXTRUDER_HH

#include "math/Vec2D.hh"
#include "MeshNode.hh"
#include "util/Log.hh"

namespace top10
{
  namespace graphX
  {

    void makeShape3d(const top10::math::Vector& pos, const top10::math::Vector& e2,
                     const std::vector< top10::math::Vec2D >& shape,
                     std::vector< top10::math::Vector>* shape_out);

    void connectShapes(const std::vector< top10::math::Vector >& shape1, double s1,
                       const std::vector< top10::math::Vector >& shape2, double s2,
		       top10::math::Mesh* mesh_out,
		       std::vector< MeshNode::TextureCoord >* texture_coords_out);

    //! Extrude a 2d shape along a 3d path.
    /*!
    Path must have methods eval(double) and eval_tg(double) returning top10::math::Vector.
    \param path The path.
    \param shape The 2d shape to extrude.
    \param step Step between points where the path is evaluated.
    */
    template< typename Path >
    MeshNode* extrude(const Path& path, const std::vector<top10::math::Vec2D>& shape, double step = 0.1)
    {
      using top10::math::Vector;

      top10::math::Mesh* mesh = new top10::math::Mesh;

      std::vector< MeshNode::TextureCoord > tex_coords;

      std::vector<Vector> prev;
      makeShape3d( path.eval(0.0), path.eval_tg(0.0), shape, &prev );

      double s = 0.0;
      for (double t=step; t<=1.0; t+=step)
      {
	const double s2 = s + (path.eval(t) - path.eval(t-step)).size();
	std::vector<Vector> curr;
	makeShape3d( path.eval(t), path.eval_tg(t), shape, &curr );

	connectShapes(prev, s, curr, s2, mesh, &tex_coords);

	prev.swap(curr);
	s = s2;
      }

      MeshNode* ret = new MeshNode(mesh, tex_coords, "Extruded mesh");
      if (ret)
	ret->setShadowVolumeFlag(false);
      return ret;
    }

  }
}

#endif
