/*
  Top10, a racing simulator
  Copyright (C) 2000-2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "Extruder.hh"

namespace top10
{
  namespace graphX
  {

    //! Generate a 3d flat shape from a 2d shape, a position and a direction.
    /*!
    \param pos Position of (0,0) in the 3d world.
    \param e2 Vector normal to the 3d flat shape, points "forward".
    \param shape The 2d shape
    \param shape_out The generated 3d flat shape is concatenated to shape_out.
    */
    void makeShape3d(const top10::math::Vector& pos, const top10::math::Vector& e2,
                     const std::vector< top10::math::Vec2D >& shape,
                     std::vector< top10::math::Vector>* shape_out)
    {
      top10::math::Vector e0 = e2 ^ top10::math::Vector(0.0, 1.0, 0.0);

      double s = e0.size();
      if (s >= SMALL_VECTOR)
	e0 /= s;
      else
	e0 = top10::math::Vector(0.0, 0.0, 1.0);

      for (unsigned int i = 0; i < shape.size(); ++i)
      {
	const top10::math::Vector p =
	  pos + e0 * shape[i].x + top10::math::Vector(0.0, shape[i].y, 0.0);
	
	shape_out->push_back(p);
      }
    }



    //! Connect two flat shapes.
    /*!
    \param shape1 The first shape
    \param s1 s texture coordinate of the vertices of shape1
    \param shape2 The second shape
    \param s2 s texture coordinate of the vertices of shape2
    \param mesh_out Output parameter, the shape where faces are added.
    \param tex_out Texture coordinates of each vertex.

    The s component of each texture coordinate is
    */
    void connectShapes(const std::vector< top10::math::Vector >& shape1, double s1, 
                       const std::vector< top10::math::Vector >& shape2, double s2,
		       top10::math::Mesh* mesh_out,
		       std::vector< MeshNode::TextureCoord >* tex_out)
    {
      using top10::math::Vector;

      assert( shape1.size() == shape2.size() );

      if (shape1.size() <= 1)
	return;

      const int n0 = mesh_out->getVertices()->size();

      Vector prev1 = shape1[0];
      Vector prev2 = shape2[0];
      double t1 = 0.0;
      double t2 = 0.0;

      mesh_out->addVertex(prev1);

      mesh_out->addVertex(prev2);

      for (unsigned int i=1; i < shape1.size(); ++i)
      {
	Vector curr1 = shape1[i];
	Vector curr2 = shape2[i];

	const double T1 = t1 + (curr1 - prev1).size();
	const double T2 = t2 + (curr2 - prev2).size();

	mesh_out->addVertex(curr1);
	mesh_out->addVertex(curr2);

	mesh_out->addFace( n0 + (i-1)*2, n0 + 1 + (i-1)*2, n0 + i*2 );
	tex_out->push_back( MeshNode::TextureCoord(s1, t1) );
	tex_out->push_back( MeshNode::TextureCoord(s2, t2) );
	tex_out->push_back( MeshNode::TextureCoord(s1, T1) );

	mesh_out->addFace( n0 + 1 + (i-1)*2, n0 + 1 + i*2, n0 + i*2 );
	tex_out->push_back( MeshNode::TextureCoord(s2, t2) );
	tex_out->push_back( MeshNode::TextureCoord(s2, T2) );
	tex_out->push_back( MeshNode::TextureCoord(s1, T1) );

	prev1 = curr1;
	prev2 = curr2;
	t1 = T1;
	t2 = T2;
      }
    }

  }
}
