/*
  Top 10, a racing simulator
  Copyright (C) 2007  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@gmail.com
*/

#include "CubeMap.hh"

#include <GL/glu.h>
#include <GL/glext.h>

#include <SDL/SDL.h>
#include <SDL/SDL_image.h>

#include <algorithm>

#include "util/Log.hh"

#define TESTGLERR(txt)	\
	{\
	int glerr = glGetError();\
	if (glerr) {\
	Log::getSingle()->send(Log::Error, "CubeMap", std::string("OpenGL error (") + std::string(txt) + std::string("): ") + std::string(reinterpret_cast<const char*>(gluErrorString(glerr))));\
	}}

namespace top10 { namespace graphX {

  CubeMap::CubeMap()
    : id(0)
  {
    glGenTextures(1, &id);
  }

  CubeMap::CubeMap(const std::string& front, const std::string& back,
		   const std::string& left, const std::string& right,
		   const std::string& top, const std::string& bottom)
		   : id (0)
  {
    glGenTextures(1, &id);

    SDL_Surface* surf = 0;

#define LOAD(front, Front)\
    surf = IMG_Load(front.c_str());\
    if (surf)\
    {\
      setFace(surf, Front);\
      SDL_FreeSurface(surf);\
    }\

    LOAD(front, Front);
    LOAD(back, Back);
    LOAD(left, Left);
    LOAD(right, Right);
    LOAD(top, Top);
    LOAD(bottom, Bottom);
#undef LOAD
  }

  CubeMap::~CubeMap()
  {
    glDeleteTextures(1, &id);
  }

  GLuint CubeMap::getId() const
  {
    return id;
  }

  void CubeMap::setFace(SDL_Surface* surface, Face face)
  {
    using top10::util::Log;

    void* buff = 0;
    int gluerr;
    SDL_Surface *image;
    SDL_Rect area;
    Uint32 saved_flags;
    Uint8  saved_alpha;

    // Convert to rgba format
    image = SDL_CreateRGBSurface(
      SDL_SWSURFACE,
      surface->w, surface->h,
      32,
#if SDL_BYTEORDER == SDL_LIL_ENDIAN /* OpenGL RGBA masks */
      0x000000FF,
      0x0000FF00,
      0x00FF0000,
      0xFF000000
#else
      0xFF000000,
      0x00FF0000,
      0x0000FF00,
      0x000000FF
#endif
      );

    if ( image == NULL )
    {
      Log::getSingle()->send(Log::Error, "CubeMap", "Failed to create surface");
      return;
    }

    /* Save the alpha blending attributes */
    saved_flags = surface->flags&(SDL_SRCALPHA|SDL_RLEACCELOK);
    saved_alpha = surface->format->alpha;
    if ( (saved_flags & SDL_SRCALPHA) == SDL_SRCALPHA ) {
      SDL_SetAlpha(surface, 0, 0);
    }

    /* Copy surface into image */
    area.x = 0;
    area.y = 0;
    area.w = surface->w;
    area.h = surface->h;
    SDL_BlitSurface(surface, &area, image, &area);

    /* Do some magic mirroring, can't quite understand why the coordinate system
       is so strange... */
    if (face == Top || face == Bottom)
    {
      // Do a vertical mirroring
      Uint32* ptr_up = (Uint32*)image->pixels;
      Uint32* ptr_down = ptr_up + image->w * (image->h -1);

      while (ptr_up < ptr_down)
      {
	std::swap_ranges(ptr_up, ptr_up + image->w, ptr_down);
	ptr_up += image->w;
	ptr_down -= image->w;
      }
    }
    else
    {
      // Do an horizontal mirror
      Uint32* left = (Uint32*)image->pixels;
      for (int i=0; i<image->h; ++i)
      {
	std::reverse(left, left+image->w);
	left += image->w;
      }
    }

    /* Restore the alpha blending attributes */
    if ( (saved_flags & SDL_SRCALPHA) == SDL_SRCALPHA ) {
      SDL_SetAlpha(surface, saved_flags, saved_alpha);
    }

    /* Bind to the cube map */
    glBindTexture(GL_TEXTURE_CUBE_MAP_EXT, id);
    TESTGLERR("bind");

    /* Select the GLenum corresponding to the face of the cube we are setting. */
    GLenum target;
    switch (face)
    {
    case Front:  target = GL_TEXTURE_CUBE_MAP_NEGATIVE_Z_EXT; break;
    case Back:   target = GL_TEXTURE_CUBE_MAP_POSITIVE_Z_EXT; break;
    case Left:   target = GL_TEXTURE_CUBE_MAP_NEGATIVE_X_EXT; break;
    case Right:  target = GL_TEXTURE_CUBE_MAP_POSITIVE_X_EXT; break;
    case Top:    target = GL_TEXTURE_CUBE_MAP_POSITIVE_Y_EXT; break;
    case Bottom: target = GL_TEXTURE_CUBE_MAP_NEGATIVE_Y_EXT; break;
    }

    /* Copy data to the texture, as a bonus resize the image to a power of 2. */
    gluBuild2DMipmaps(target, GL_RGBA, image->w, image->h, GL_RGBA, GL_UNSIGNED_BYTE, image->pixels);

    /* Temp SDL_Surface no longer needed */
    SDL_FreeSurface(image); 
  }

}}