/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include <vector>
#include <cmath>
#include <cassert>
#include <algorithm>

#include "Triangle.hh"
#include "TrackSide.hh"
#include "util/PathFinder.hh"

//SDL Stuff
#include <SDL.h>
#include <SDL_image.h>
extern "C" {
#include "extras/sdl2gl.h"
}

using std::vector;
using std::ceil;
using std::cerr;
using std::endl;

using top10::math::Vector;
using top10::ui_interactive::TriangleSet;

TriangleSet::texture_names_map TriangleSet::texture_names;

TriangleSet::TriangleSet(top10::helpers::PolygonSet* polys):
  polygons(polys),
  listname(0),
  octree(0)
{
  // Load all necessary textures
  vector<std::string>::const_iterator surf_name = polys->surf_names.begin();
  for (vector<top10::helpers::Surface>::const_iterator surface = polys->surfaces.begin();
       surface != polys->surfaces.end();
       ++surface, ++surf_name) {
    assert(surf_name != polys->surf_names.end());

    Texture new_texture;
    new_texture.tex_used = false;

    if (!surface->texture_filename.empty()) {
      texture_names_map::const_iterator f = texture_names.find(surface->texture_filename);
      GLuint sdl_tex;
      if (f == texture_names.end()) {
	std::string path = top10::util::PathFinder::find(surface->texture_filename);
	if (path.empty()) {
	  std::cerr<<"Could not find "<<surface->texture_filename<<std::endl;
	  continue;
	}
	
	SDL_Surface* sdl_surf = IMG_Load(path.c_str());
	
	if (!sdl_surf) {
	  std::cerr<<"Could not load "<<path<<std::endl;
	  continue;
	}
	
	sdl_tex = createGLfromSDL(sdl_surf, 0, 0);   //Texture size not specified. Chosen to fit best original image.
	TriangleSet::setTexture(surface->texture_filename, sdl_tex);
	
	std::cout<<"Loaded texture "<<(*surf_name)<<" from "<<path<<std::endl;
      }
      else {
	sdl_tex = f->second;
      }
      new_texture.tex_used = true;
      new_texture.tex_name = sdl_tex;
      new_texture.alt_used = false;
    }

    textures.push_back(new_texture);
  }
}

GLuint TriangleSet::findTexture(std::string name)
{
  texture_names_map::const_iterator p = texture_names.find(name);
  if (p == texture_names.end()) throw top10::util::Error("Texture "+name+" not found");
  return p->second;
}

void TriangleSet::setTexture(std::string name, GLuint tex)
{
  texture_names[name] = tex;
}

void TriangleSet::buildOctree() const
{
  if (octree) delete octree;
  octree = new top10::physX::Octree(polygons);
}

void TriangleSet::drawGL(const Frustum& frustum) const
{
  if (octree == 0) buildOctree();

  const top10::physX::Octree::ShapeVec* polys = octree->getShapeVec();
  top10::physX::Octree::ShapeRefs refs = octree->getVolume(frustum, 4);
  std::sort(refs.begin(), refs.end());

  int old_ref=-1;
  for (top10::physX::Octree::ShapeRefs::const_iterator ref = refs.begin(); ref != refs.end(); ++ref) {
    if (*ref == old_ref) continue;
    else old_ref = *ref;

    const top10::helpers::Polygon& polygon = polys->at(*ref);
	
    // Skip points and segments
    if (polygon.p.size() < 3) continue;

    // Enable texturing
    if (textures[polygon.surface_name].tex_used && !textures[polygon.surface_name].alt_used) {
      glEnable(GL_TEXTURE_2D);
      glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
      glBindTexture(GL_TEXTURE_2D, textures[polygon.surface_name].tex_name);
    }
	
    float color[4];
    if (textures[polygon.surface_name].alt_used) {
      color[0] = textures[polygon.surface_name].r/255.0;
      color[1] = textures[polygon.surface_name].g/255.0;
      color[2] = textures[polygon.surface_name].b/255.0;
      color[3] = 1.0;
    }
    else {
      color[0] = polygons->surfaces[polygon.surface_name].r/255.0;
      color[1] = polygons->surfaces[polygon.surface_name].g/255.0;
      color[2] = polygons->surfaces[polygon.surface_name].b/255.0;
      color[3] = polygons->surfaces[polygon.surface_name].a/255.0;
    }

    glColor4fv(color);
    glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, color);

    //TODO: smooth normals using neighbours
    Vector normal = ((polygon.p[1] - polygon.p[0])^(polygon.p[2] - polygon.p[0]));
    double size = normal.size();
    if (size > SMALL_VECTOR) {
      normal/=size;
      glNormal3f(normal.x, normal.y, normal.z);
	   
      glBegin(GL_POLYGON);
	  
      int st_idx = 0;
      for (std::vector<Vector>::const_iterator v = polygon.p.begin();
	   v != polygon.p.end();
	   ++v, ++st_idx) {
	if (textures[polygon.surface_name].tex_used && !textures[polygon.surface_name].alt_used)
	  glTexCoord2f(polygon.texels[st_idx].x, polygon.texels[st_idx].y);
	glVertex3f(v->x, v->y, v->z);
      }
	  
      glEnd();
	  
    } // if SMALL_VECTOR
    glDisable(GL_TEXTURE_2D);
  }
}

void TriangleSet::drawPartGL(int idx, bool wireframe, float alpha, bool use_alt) const
{
  const top10::helpers::Mesh& mesh(polygons->meshes[idx]);

  // Skip each hidden mesh
  if (mesh.hidden) return;

  // Loop over polygons in this mesh
  for (int poly_idx = mesh.begin;
       poly_idx != mesh.end;
       ++poly_idx) {
    const top10::helpers::Polygon& polygon = polygons->at(poly_idx);
	
    // Skip points and segments
    if (polygon.p.size() < 3) continue;

    // Enable texturing
    if (!wireframe && !use_alt && textures[polygon.surface_name].tex_used && !textures[polygon.surface_name].alt_used) {
      glEnable(GL_TEXTURE_2D);
      glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
      glBindTexture(GL_TEXTURE_2D, textures[polygon.surface_name].tex_name);
    }
	
    float color[4];
    if (use_alt || textures[polygon.surface_name].alt_used) {
      color[0] = textures[polygon.surface_name].r/255.0;
      color[1] = textures[polygon.surface_name].g/255.0;
      color[2] = textures[polygon.surface_name].b/255.0;
      color[3] = alpha;
    }
    else {
      color[0] = polygons->surfaces[polygon.surface_name].r/255.0;
      color[1] = polygons->surfaces[polygon.surface_name].g/255.0;
      color[2] = polygons->surfaces[polygon.surface_name].b/255.0;
      color[3] = alpha*polygons->surfaces[polygon.surface_name].a/255.0;

    }

    glColor4fv(color);
    glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, color);

    //TODO: smooth normals using neighbours
    Vector normal = ((polygon.p[1] - polygon.p[0])^(polygon.p[2] - polygon.p[0]));
    double size = normal.size();
    if (size > SMALL_VECTOR) {
      normal/=size;
      glNormal3f(normal.x, normal.y, normal.z);

      if (!wireframe) glBegin(GL_POLYGON);
      else glBegin(GL_LINE_LOOP);
	  
      int st_idx = 0;
      for (std::vector<Vector>::const_iterator v = polygon.p.begin();
	   v != polygon.p.end();
	   ++v, ++st_idx) {
	if (!wireframe && textures[polygon.surface_name].tex_used && !textures[polygon.surface_name].alt_used)
	  glTexCoord2f(polygon.texels[st_idx].x, polygon.texels[st_idx].y);
	glVertex3f(v->x, v->y, v->z);
      }
	  
      glEnd();
	  
    } // if SMALL_VECTOR
    glDisable(GL_TEXTURE_2D);
  } // for poly_idx
}

void TriangleSet::drawGL() const
{
  if (listname == 0) {
    listname = glGenLists(1);
    glNewList(listname, GL_COMPILE);

    for (int idx=0;
	 idx < polygons->meshes.size();
	 ++idx) {
      drawPartGL(idx);
    } // for idx
    glEndList();
  } // if listname
  else {
    glCallList(listname);
  }
}

void TriangleSet::invalidateListNames() const
{
  if (listname) glDeleteLists(listname, 1);
  listname = 0;
}

TriangleSet::~TriangleSet()
{
  invalidateListNames();
  if (octree) delete octree;
}

#include "helpers/GenericOctree-template.cpp"
