/*
  Top10, a racing simulator
  Copyright (C) 2000-2004  Johann Deneux
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
  Authors can be contacted at following electronic addresses:
  Johann Deneux: johann.deneux@it.uu.se
*/

#include <fstream>

#include <cmath>

#include "Kart.hh"
#include "helpers/3dsRead.hh"
#include "util/error.hh"
#include "util/PathFinder.hh"
#include "util/Parse.hh"
#include "track/Track.hh"
#include "math/Matrix.hh"

using namespace top10::ui_interactive;
using namespace top10::math;
using namespace std;

using top10::helpers::Read3DS;
using top10::util::Error;

const double Kart::memory_factor = 0.5;

static inline void glVertex(Vector v)
{
  glVertex3f(v.x, v.y, v.z);
}

Kart::Kart(top10::track::Track* track, Vector _center, Vector e1,
	   const KartGL* kgl):
  top10::physX::Kart(track, _center, e1),
  friction_left_accum(0.0),
  friction_right_accum(0.0),
  side_force_accum(0.0),
  friction_force_fl(0,0,0),
  friction_force_fr(0,0,0),
  friction_force_rl(0,0,0),
  friction_force_rr(0,0,0),
  kart_gl(kgl)
{
}

void Kart::updateUI(double)
{
  friction_left_accum = (memory_factor * friction_left_accum
			 + wheel_rl.getFriction())
    / (1+memory_factor);
  friction_right_accum = (memory_factor * friction_right_accum
			  + wheel_rr.getFriction())
    / (1+memory_factor);
  double mom_mem = 0.9;
  double mom_new = 0.1;
  //  side_force_accum = (mom_mem * side_force_accum + mom_new * body.getForce() * right) / (mom_new + mom_mem);

  friction_force_fl = (friction_force_fl * memory_factor + wheel_fl.getFrictionForce());
  friction_force_fr = (friction_force_fr * memory_factor + wheel_fr.getFrictionForce());
  friction_force_rl = (friction_force_rl * memory_factor + wheel_rl.getFrictionForce());
  friction_force_rr = (friction_force_rr * memory_factor + wheel_rr.getFrictionForce());
}

double Kart::getLeftFriction() const
{
  return friction_left_accum;
}

double Kart::getRightFriction() const
{
  return friction_right_accum;
}

Frustum Kart::getCamera(camEnum cam) const
{
  const top10::math::Matrix3& orient(body.getOrientation());
  Vector right = orient * Vector(0,0,1);
  Vector forward = orient * Vector(1,0,0);
  Vector up = orient * Vector(0,1,0);
  Vector center = body.getMassCenter();
  double fov = 30;

  Vector pos, dir, dup;
  switch(cam) {
  case In:
    pos = center - forward*0.2 + Vector(0,0.4,0);
    dir = forward - up*0.2;
    dup = Vector(0,1,0);
    break;
  case Fore:
    pos = center + forward*5 + Vector(0,1,0);
    dir = -forward - Vector(0,0.2,0);;
    dup = Vector(0,1,0);
    fov = 50;
    break;
  case Back:
    pos = center - forward*5 + Vector(0,1.5,0);
    dir = forward - Vector(0,0.2,0);
    dup = Vector(0,1,0);
    fov = 50;
    break;
  case Left:
    pos = center - right*15 + Vector(0,1,0);
    dir = right;
    dup = Vector(0,1,0);
    fov = 50;
    break;
  case Right:
    pos = center + right*15 + Vector(0,1,0);
    dir = -right;
    dup = Vector(0,1,0);
    fov = 50;
    break;
  case Top:
    pos = center + Vector(0,50,0);
    dir = Vector(0,-1,0);
    fov = 50;
    dup = forward;
    break;
  default: abort();
  };

  return Frustum(pos, dir, dup, 1.4, 30, 0.5, 500);
}

top10::racing::KartState Kart::getKartState() const
{
  top10::racing::KartState ret;
  ret.timestamp = 0;
  ret.orient = body.getOrientation();
  ret.translation = body.getTranslation();
  ret.steer = steer_angle;

  return ret;
}

void Kart::drawGL() const
{
  if (!kart_gl) return;
  kart_gl->drawGL(body.getTranslation(), body.getMassCenterL(),
		  top10::math::Matrix<3>(body.getOrientation()), steer_angle,
		  wheel_fl.getAngle(), wheel_fr.getAngle(),
		  wheel_fl.getHeight(), wheel_fr.getHeight(),
		  wheel_rl.getHeight(), wheel_rr.getHeight());
}

void Kart::drawGL(const Frustum& clip) const
{
  if (!kart_gl) return;

  kart_gl->drawGL(body.getTranslation(), body.getMassCenterL(),
		  top10::math::Matrix<3>(body.getOrientation()), steer_angle,
		  wheel_fl.getAngle(), wheel_fr.getAngle(),
		  wheel_fl.getHeight(), wheel_fr.getHeight(),
		  wheel_rl.getHeight(), wheel_rr.getHeight(),
		  clip);
}

void KartGL::drawGL(Vector translation, Vector mass_center_L,
		    top10::math::Matrix3 orient, double steer_angle,
		    double steer_fl, double steer_fr,
		    double h_fl, double h_fr,
		    double h_rl, double h_rr,
		    const Frustum& clip) const
{
  top10::math::Box box = bounding_box;
  box.rotate(orient);
  box.translate(translation + mass_center_L + Vector(0,fl.y,0));
  if (clip.overlap(box) == top10::helpers::None) return;
  else drawGL(translation, mass_center_L, orient, steer_angle, steer_fl, steer_fr, h_fl, h_fr, h_rl, h_rr);
}

void KartGL::drawGL(Vector translation, Vector mass_center_L,
		    top10::math::Matrix3 orient, double steer_angle,
		    double steer_fl, double steer_fr,
		    double h_fl, double h_fr,
		    double h_rl, double h_rr) const
{
  glPushMatrix();

  // Translation
  Vector t = translation;
  Vector center = mass_center_L;
  glTranslatef(t.x+center.x, fl.y + t.y+center.y, t.z+center.z);

  // Orientation
  GLdouble orient_gl[4][4];
  OrthoNorm3 M3 = orient;
  Matrix4 M4(M3);
  M4.toGL(orient_gl);
  glMultMatrixd((GLdouble*)orient_gl);

  // Rotate around the center of mass
  glTranslatef(-center.x, -center.y, -center.z);

  // Draw the body
  glPushMatrix();
  transforms[BODY].toGL(orient_gl);
  glMultMatrixd((GLdouble*)orient_gl);
  models[BODY]->drawPartGL(mesh_idxs[BODY]);
  glPopMatrix();

  // Draw the steering wheel
  glPushMatrix();
  transforms[STEER].toGL(orient_gl);
  glTranslated(steer_center.x, steer_center.y, steer_center.z);
  glRotated(540.0*steer_angle/M_PI, steer_axis.x, steer_axis.y, steer_axis.z);
  glTranslated(-steer_center.x, -steer_center.y, -steer_center.z);
  glMultMatrixd((GLdouble*)orient_gl);
  models[STEER]->drawPartGL(mesh_idxs[STEER]);
  glPopMatrix();

  // Draw the wheels
  // front wheels
  glPushMatrix();
  transforms[WHEEL_FL].toGL(orient_gl);
  glTranslated(0, h_fl, 0);
  glTranslated(fl.x, fl.y, fl.z);
  glRotated(180.0*steer_fl/M_PI, 0, 1, 0);
  glTranslated(-fl.x, -fl.y, -fl.z);
  glMultMatrixd((GLdouble*)orient_gl);
  models[WHEEL_FL]->drawPartGL(mesh_idxs[WHEEL_FL]);
  glPopMatrix();

  glPushMatrix();
  transforms[WHEEL_FR].toGL(orient_gl);
  glTranslated(0, h_fr, 0);
  glTranslated(fr.x, fr.y, fr.z);
  glRotated(180.0*steer_fr/M_PI, 0, 1, 0);
  glTranslated(-fr.x, -fr.y, -fr.z);
  glMultMatrixd((GLdouble*)orient_gl);
  models[WHEEL_FR]->drawPartGL(mesh_idxs[WHEEL_FR]);
  glPopMatrix();

  // back wheels
  glPushMatrix();
  transforms[WHEEL_RL].toGL(orient_gl);
  glTranslated(0, h_rl, 0);
  glMultMatrixd((GLdouble*)orient_gl);
  models[WHEEL_RL]->drawPartGL(mesh_idxs[WHEEL_RL]);
  glPopMatrix();

  glPushMatrix();
  transforms[WHEEL_RR].toGL(orient_gl);
  glTranslated(0, h_rr, 0);
  glMultMatrixd((GLdouble*)orient_gl);
  models[WHEEL_RR]->drawPartGL(mesh_idxs[WHEEL_RR]);
  glPopMatrix();
  
  glPopMatrix();
}

#define EXPECT(str) if (token != str) throw Error("Expected "+std::string(str)+", got "+token)

void KartGL::parsePart(std::istream& in, const char* keyword,
		       std::string* alias,
		       std::string* object, top10::math::Matrix4* M)
{
  using top10::util::parseString;

  std::string token;

  in>>token;
  EXPECT(keyword);
  in>>token;
  EXPECT("MODEL");
  *alias = parseString(in);
  *object = parseString(in);
  in>>token;
  EXPECT("TRANSFORM");
  in>>(*M);
}

KartGL::KartGL(std::string model_filename):
  bounding_box(Vector(0,0,0), Vector(2,0,0), Vector(0,2,0), Vector(0,0,2))
{
  std::string path = top10::util::PathFinder::find(model_filename);
  if (path.empty()) throw Error("Could not find ") + model_filename;
  std::ifstream in(path.c_str());
  if (!in) throw Error("Could not open "+path);

  /* Expected:
     BEGIN MODELS
     <model filename> <alias name>
     ...
     END
  */

  std::string token;
  in>>token;
  EXPECT("BEGIN");
  in>>token;
  EXPECT("MODELS");
  for (;;) {
    token = top10::util::parseString(in);
    if (token == "END") break;
    if (in.eof()) throw Error("Early EOF");

    path = top10::util::PathFinder::find(token);
    if (path.empty()) throw Error("Could not find "+token);
    top10::helpers::PolygonSet* model = top10::helpers::PolygonSet::loadNew(path);
    if (!model) throw Error("Could not load "+path);

    token = top10::util::parseString(in);
    model_map[token] = new TriangleSet(model);
  }

  /* Expected:
     BEGIN BODY
     <alias> <object name>
     TRANSFORM <4x4 matrix (16 floats)>
     END
  */
  std::map<std::string, TriangleSet*>::iterator f;

  const char* keywords[] = {"BODY", "WHEEL_FL", "WHEEL_FR", "WHEEL_RL", "WHEEL_RR", "STEERING_WHEEL"};
  std::string aliases[6];
  std::string objects[6];
  for (int i=0; i<6; ++i) {
    in>>token;
    EXPECT("BEGIN");
    parsePart(in, keywords[i], &(aliases[i]), &(objects[i]), &(transforms[i]));
    if (i == STEER) {
      in>>token;
      EXPECT("CENTER");
      in>>steer_center.x;
      in>>steer_center.y;
      in>>steer_center.z;

      in>>token;
      EXPECT("AXIS");
      in>>steer_axis.x;
      in>>steer_axis.y;
      in>>steer_axis.z;
      double axis_size = steer_axis.size();
      if (axis_size < SMALL_VECTOR) throw Error("Axis of steering wheel is too small");
      steer_axis /= axis_size;
    }
    else if (i == WHEEL_FL) {
      in>>token;
      EXPECT("CENTER");
      in>>fl.x>>fl.y>>fl.z;
      std::cout<<"FL:"<<fl<<std::endl;
    }
    else if (i == WHEEL_FR) {
      in>>token;
      EXPECT("CENTER");
      in>>fr.x>>fr.y>>fr.z;
      std::cout<<"FR:"<<fr<<std::endl;
    }
    else if (i == WHEEL_RL) {
      in>>token;
      EXPECT("CENTER");
      in>>rl.x>>rl.y>>rl.z;
    }
    else if (i == WHEEL_RR) {
      in>>token;
      EXPECT("CENTER");
      in>>rr.x>>rr.y>>rr.z;
    }

    in>>token;
    EXPECT("END");

    f = model_map.find(aliases[i]);
    if (f == model_map.end()) throw Error("Bad alias "+aliases[i]);
    models[i] = f->second;
    mesh_idxs[i] = f->second->polygons->findMeshIndex(objects[i]);
  }
}
